package ru.yandex.stockpile.server.shard.load;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicReference;

import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryWithStats;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;


/**
 * <p>This class allows to traverse over specific range of files in KV tablet.
 *
 * <p>Because files are traversed sequentially this implementation uses sequential
 * prefetch technique to pipeline load of the next files (async network operation)
 * and their processing (CPU intensive operation).
 *
 * @author Sergey Polovko
 */
public abstract class KvReadRangeIterator implements AsyncIterator<KvEntryWithStats> {

    private final NameRange nameRange;

    private volatile CompletableFuture<KvReadRangeResult> asyncRead;
    private final AtomicReference<LocalState> state = new AtomicReference<>();

    public KvReadRangeIterator(NameRange nameRange) {
        this.nameRange = nameRange;
        this.asyncRead = readNext(nameRange);
    }

    /**
     * Load range of KV files.
     *
     * Actual communication with Kikimr must be implemented in subclasses. It allows
     * to reuse code of retrying async operations implemented in ShardProcess.
     */
    protected abstract CompletableFuture<KvReadRangeResult> readNext(NameRange nameRange);

    @Override
    public CompletableFuture<KvEntryWithStats> next() {
        while (true) {
            LocalState state = this.state.get();
            if (state != null) {
                if (state.isEmpty()) {
                    // done
                    return CompletableFuture.completedFuture(null);
                }

                KvEntryWithStats next = state.next();
                if (next != null) {
                    return CompletableFuture.completedFuture(next);
                }

                this.state.compareAndSet(state, null);
            }

            if (CompletableFutures.isCompletedSuccessfully(asyncRead)) {
                processResult(asyncRead.getNow(null));
            } else {
                return asyncRead.thenCompose(result -> {
                    processResult(result);
                    return next();
                });
            }
        }
    }

    private void processResult(KvReadRangeResult result) {
        LocalState nextState = new LocalState(result.getEntries());
        if (this.state.compareAndSet(null, nextState)) {
            if (result.isOverrun()) {
                NameRange nextNameRange = NameRange.rangeFrom(result.getOverrunLastEntryName(), false);
                if (nameRange.getEnd() != null) {
                    nextNameRange = nextNameRange.to(nameRange.getEnd(), nameRange.isEndInclusive());
                }
                asyncRead = readNext(nextNameRange);
            } else {
                asyncRead = CompletableFuture.completedFuture(new KvReadRangeResult(new KvEntryWithStats[0], false));
            }
        }
    }

    /**
     * LOCAL STATE
     */
    private static final class LocalState {
        private static final AtomicIntegerFieldUpdater<LocalState> indexUpdater =
            AtomicIntegerFieldUpdater.newUpdater(LocalState.class, "index");

        private final KvEntryWithStats[] entries;
        private volatile int index = 0;

        LocalState(KvEntryWithStats[] entries) {
            this.entries = entries;
        }

        boolean isEmpty() {
            return entries.length == 0;
        }

        KvEntryWithStats next() {
            int index = indexUpdater.getAndIncrement(this);
            if (index >= entries.length) {
                return null;
            }

            KvEntryWithStats value = entries[index];
            entries[index] = null; // faster memory reclamation
            return value;
        }
    }
}
