package ru.yandex.stockpile.server.shard.load;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;

import javax.annotation.WillClose;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunnerImpl;
import ru.yandex.misc.actor.Tasks;
import ru.yandex.misc.lang.IntegerUtils;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.config.protobuf.stockpile.EInvalidArchiveStrategy;
import ru.yandex.solomon.util.concurrent.CountDownCallback;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricToArchiveMap;
import ru.yandex.stockpile.server.shard.InvalidArchiveStrategy;
import ru.yandex.stockpile.server.shard.MetricArchives;


/**
 * @author Sergey Polovko
 */
public class MetricArchivesMerger {
    private static final Logger logger = LoggerFactory.getLogger(MetricArchivesMerger.class);
    private final Merger[] mergers;

    public MetricArchivesMerger(int shradId, InvalidArchiveStrategy invalidArchiveStrategy, Executor executor, int mergersCount) {
        this.mergers = new Merger[IntegerUtils.roundUpToPowerOf2(mergersCount)];
        for (int i = 0; i < mergers.length; i++) {
            mergers[i] = new Merger(shradId, invalidArchiveStrategy, executor);
        }
    }

    public void merge(@WillClose Long2ObjectMap<MetricArchiveImmutable> metrics, CountDownCallback callback) {
        try {
            for (Long2ObjectMap.Entry<MetricArchiveImmutable> e : metrics.long2ObjectEntrySet()) {
                Merger merger = getMerger(e.getLongKey());
                merger.enqueue(e.getLongKey(), e.getValue(), callback);
            }
        } catch (Throwable t) {
            callback.onFailure(t);
        }
    }

    public CompletableFuture<Void> merge(@WillClose Long2ObjectMap<MetricArchiveImmutable> metrics) {
        CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        merge(metrics, new CountDownCallback(metrics.size()) {
            @Override
            protected void onComplete() {
                doneFuture.complete(null);
            }

            @Override
            public void onFailure(Throwable t) {
                doneFuture.completeExceptionally(t);
            }
        });
        return doneFuture;
    }

    public MetricToArchiveMap combineResults() {
        int metricsCount = 0;
        for (Merger m : mergers) {
            metricsCount += m.getResult().size();
        }

        Long2ObjectOpenHashMap<MetricArchiveMutable> result = new Long2ObjectOpenHashMap<>(metricsCount);
        long memorySize = 0;
        for (Merger merger : mergers) {
            result.putAll(merger.getResult());
            for (MetricArchiveMutable archive : merger.getResult().values()) {
                memorySize += archive.memorySizeIncludingSelf();
            }
        }

        return new MetricToArchiveMap(result, memorySize);
    }

    private Merger getMerger(long localId) {
        final int index = Long.hashCode(localId) & (mergers.length - 1);
        return mergers[index];
    }

    /**
     * Merger queue item.
     */
    private static final class Item {
        final long localId;
        final MetricArchiveImmutable metric;
        final CountDownCallback callback;

        Item(long localId, MetricArchiveImmutable metric, CountDownCallback callback) {
            this.localId = localId;
            this.metric = metric;
            this.callback = callback;
        }
    }

    /**
     * Merger.
     */
    private static final class Merger implements Runnable {
        private final int shardId;
        private final InvalidArchiveStrategy invalidArchiveStrategy;
        private final Executor executor;
        private final Tasks tasks = new Tasks();
        private final ConcurrentLinkedQueue<Item> queue = new ConcurrentLinkedQueue<>();
        private final Long2ObjectOpenHashMap<MetricArchiveMutable> result = new Long2ObjectOpenHashMap<>();

        Merger(int shardId, InvalidArchiveStrategy invalidArchiveStrategy, Executor executor) {
            this.shardId = shardId;
            this.invalidArchiveStrategy = invalidArchiveStrategy;
            this.executor = executor;
        }

        void enqueue(long localId, @WillClose MetricArchiveImmutable metric, CountDownCallback callback) {
            queue.offer(new Item(localId, metric, callback));
            if (tasks.addTask()) {
                ActorRunnerImpl.schedule(executor, this);
            }
        }

        Long2ObjectOpenHashMap<MetricArchiveMutable> getResult() {
            return result;
        }

        @Override
        public void run() {
            while (tasks.fetchTask()) {
                Item item;
                while ((item = queue.poll()) != null) {
                    try {
                        MetricArchiveMutable mergedArchive = result.get(item.localId);
                        if (mergedArchive == null) {
                            mergedArchive = new MetricArchiveMutable(item.metric.header());
                            result.put(item.localId, mergedArchive);
                        }
                        MetricArchives.typeSafeAppend(shardId, item.localId, mergedArchive, item.metric);
                        item.callback.countDown();
                    } catch (Throwable t) {
                        logger.error("Failed to merge archive {}/{} {}", shardId, StockpileLocalId.toString(item.localId), item.metric.header(), t);
                        if (invalidArchiveStrategy.strategy == EInvalidArchiveStrategy.DROP) {
                            item.callback.countDown();
                        } else {
                            item.callback.onFailure(t);
                        }
                    } finally {
                        item.metric.close();
                    }
                }
            }
        }
    }
}
