package ru.yandex.stockpile.server.shard.merge;

import javax.annotation.Nullable;

import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.frames.Frame;
import ru.yandex.solomon.codec.compress.frames.TimeSeriesFrameIterator;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class ArchiveItemIterator implements Iterator {
    private final StockpileFormat format;
    private final MetricType type;
    private final int mask;
    private final BitBuf compressed;
    private final int totalRecords;
    private final long lastTsMillis;
    private final TimeSeriesFrameIterator it;
    private final Frame frame;
    private int leftRecords;

    private ArchiveItemIterator(StockpileFormat format, MetricType type, int mask, BitBuf compressed, int records, long lastTsMillis) {
        this.format = format;
        this.type = type;
        this.mask = mask;
        this.compressed = compressed;
        this.totalRecords = records;
        this.leftRecords = records;
        this.lastTsMillis = lastTsMillis;
        this.it = CompressStreamFactory.createFrameIterator(compressed.asReadOnly());
        this.frame = new Frame();
    }

    public static Iterator of(MetricArchiveMutable archive) {
        if (archive.getRecordCount() == 0) {
            return EmptyIterator.INSTANCE;
        }

        archive.sortAndMerge();
        return new ArchiveItemIterator(archive.getFormat(), archive.getType(), archive.columnSetMask(), archive.getCompressedDataRaw(), archive.getRecordCount(), archive.getLastTsMillis());
    }

    public static Iterator of(MetricArchiveImmutable archive, long lastTsMillis) {
        if (archive.getRecordCount() == 0) {
            return EmptyIterator.INSTANCE;
        }

        return new ArchiveItemIterator(archive.getFormat(), archive.getType(), archive.columnSetMask(), archive.getCompressedDataRaw(), archive.getRecordCount(), lastTsMillis);
    }

    @Override
    public MetricType type() {
        return type;
    }

    @Override
    public int columnSetMask() {
        return mask;
    }

    @Override
    public int elapsedRecords() {
        return totalRecords;
    }

    @Nullable
    @Override
    public Item next() {
        if (!it.next(frame)) {
            return null;
        }

        var compressedFrame = compressed.slice(frame.pos, frame.size);
        if (frame.closed) {
            leftRecords -= frame.records;
            return new ItemFrame(
                format,
                type,
                mask,
                frame.records,
                frame.firstTsMillis,
                frame.lastTsMillis,
                compressedFrame
            );
        }

        var it = iterator(compressedFrame, leftRecords);
        return new ItemIterator(it, firstTsMillis(frame, compressedFrame.asReadOnly()), lastTsMillis, compressedFrame.bytesSize());
    }

    private long firstTsMillis(Frame frame, BitBuf buffer) {
        if (frame.firstTsMillis != 0) {
            return frame.firstTsMillis;
        }

        var it = iterator(buffer, 1);
        var point = RecyclableAggrPoint.newInstance();
        try {
            it.next(point);
            return point.tsMillis;
        } finally {
            point.recycle();
        }
    }

    private AggrGraphDataListIterator iterator(BitBuf buffer, int records) {
        return new BinaryAggrGraphDataListIterator(type, mask, buffer, records);
    }
}
