package ru.yandex.stockpile.server.shard.merge;

import java.time.Instant;

import javax.annotation.Nullable;

import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.bits.BitBufAllocator;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class CompressCollector implements ItemVisitor {
    private final StockpileFormat format;
    private final MetricType type;
    private final int mask;

    private BitBuf buffer;
    @Nullable
    private TimeSeriesOutputStream openFrame;
    private int records;
    private long lastTsMillis;
    private long frameCount;

    CompressCollector(StockpileFormat format, MetricType type, int mask, BitBuf buffer) {
        this.format = format;
        this.type = type;
        this.mask = mask;
        this.buffer = buffer;
    }

    public static CompressResult collect(StockpileFormat format, MetricType type, int mask, int elapsedBytes, Iterator it) {
        BitBuf buffer = BitBufAllocator.buffer(elapsedBytes);
        CompressCollector collector = new CompressCollector(format, type, mask, buffer);
        Item item;
        while ((item = it.next()) != null) {
            item.visit(collector);
        }
        return collector.finish();
    }

    public CompressResult finish() {
        return new CompressResult(format, type, mask, getBuffer(), records, lastTsMillis, frameCount);
    }

    public BitBuf getBuffer() {
        appendOpenFrame();
        return buffer;
    }

    @Override
    public void visit(ItemFrame item) {
        if (item.getType() != type) {
            throw new IllegalStateException("Expected type: " + type + " but was " + item.getType());
        }

        if (records > 0 && item.getFirstTsMillis() <= lastTsMillis) {
            throw new IllegalStateException("First ts at frame  " + item + " <= then last appended " + Instant.ofEpochMilli(lastTsMillis));
        }

        if (isAbleAppendAsFrame(item)) {
            appendFrame(item.getBuffer());
            this.frameCount++;
            this.records += item.getRecord();
            this.lastTsMillis = item.getLastTsMillis();
        } else {
            appendIterator(item.iterator(), item.getElapsedBytes());
        }
    }

    @Override
    public void visit(ItemIterator item) {
        if (records > 0 && item.getFirstTsMillis() <= lastTsMillis) {
            throw new IllegalStateException("First ts at frame  " + item + " <= then last appended " + Instant.ofEpochMilli(lastTsMillis));
        }

        appendIterator(item.iterator(), item.getElapsedBytes());
    }

    public void appendIterator(AggrGraphDataListIterator iterator, int elapsedBytes) {
        if (openFrame == null) {
            openFrame = CompressStreamFactory.createOutputStream(type, mask);
        }

        openFrame.ensureCapacity(elapsedBytes);
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (iterator.next(point)) {
                openFrame.writePoint(mask, point);
                lastTsMillis = point.tsMillis;
            }
        } finally {
            point.recycle();
        }

        if (openFrame.closeFrame()) {
            frameCount++;
            appendOpenFrame();
        }
    }

    public void appendPoint(AggrPoint point) {
        if (openFrame == null) {
            openFrame = CompressStreamFactory.createOutputStream(type, mask);
        }

        openFrame.writePoint(mask, point);
        lastTsMillis = point.tsMillis;
    }

    private void appendOpenFrame() {
        if (openFrame != null) {
            appendFrame(openFrame.getCompressedData());
            records += openFrame.recordCount();
            openFrame.close();
            openFrame = null;
        }
    }

    private void appendFrame(BitBuf frame) {
        this.buffer.writeBits(frame);
    }

    private boolean isAbleAppendAsFrame(ItemFrame item) {
        if (openFrame != null) {
            return false;
        }

        if (item.getFormat() != format) {
            return false;
        }

        return item.getMask() == mask;
    }
}
