package ru.yandex.stockpile.server.shard.merge;

import java.time.Instant;
import java.util.List;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
public class ConcatIterator implements Iterator {
    private static final Logger logger = LoggerFactory.getLogger(ConcatIterator.class);
    private final List<? extends Iterator> iterators;
    private int pos;
    private long lastTsMillis;

    private ConcatIterator(List<? extends Iterator> iterators) {
        this.iterators = iterators;
    }

    public static Iterator of(List<? extends Iterator> iterators) {
        if (iterators.isEmpty()) {
            return EmptyIterator.INSTANCE;
        }

        if (iterators.size() == 1) {
            return iterators.get(0);
        }

        return new ConcatIterator(iterators);
    }

    @Override
    public MetricType type() {
        for (var it : iterators) {
            if (it.type() != MetricType.METRIC_TYPE_UNSPECIFIED) {
                return it.type();
            }
        }
        return MetricType.METRIC_TYPE_UNSPECIFIED;
    }

    @Override
    public int columnSetMask() {
        int mask = 0;
        for (var it : iterators) {
            mask |= it.columnSetMask();
        }
        return mask;
    }

    @Override
    public int elapsedRecords() {
        int records = 0;
        for (var it : iterators) {
            records += it.elapsedRecords();
        }
        return records;
    }

    @Nullable
    @Override
    public Item next() {
        for (int index = pos; index < iterators.size(); index++) {
            var it = iterators.get(index);
            var result = it.next();
            if (result == null) {
                pos++;
                continue;
            }

            if (lastTsMillis != 0 && result.getFirstTsMillis() <= lastTsMillis) {
                throw new IllegalStateException("First ts at frame  " + result + " <= then last ts from prev frame " + Instant.ofEpochMilli(lastTsMillis));
            }

            lastTsMillis = result.getLastTsMillis();
            return result;
        }

        return null;
    }
}
