package ru.yandex.stockpile.server.shard.merge;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.ConcatAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.decim.DecimPolicy;
import ru.yandex.solomon.model.timeseries.decim.DecimatingAggrGraphDataIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class DecimIterator implements Iterator {
    private final Iterator it;
    private final DecimPolicy policy;
    private final long now;
    private final long decimFromTsMillis;
    private final long decimBeforeTsMillis;
    private Item last;

    private DecimIterator(Iterator it, DecimPolicy policy, long now, long decimatedAt) {
        this.it = it;
        this.policy = policy;
        this.now = now;
        this.decimFromTsMillis = policy.getDecimFrom(decimatedAt);
        this.decimBeforeTsMillis = policy.getDecimBefore(now);
    }

    public static Iterator of(Iterator it, DecimPolicy policy, long now, long decimatedAt) {
        if (policy.isEmpty()) {
            return it;
        }

        return new DecimIterator(it, policy, now, decimatedAt);
    }

    @Override
    public MetricType type() {
        return it.type();
    }

    @Override
    public int columnSetMask() {
        return it.columnSetMask();
    }

    @Override
    public int elapsedRecords() {
        return it.elapsedRecords();
    }

    @Nullable
    @Override
    public Item next() {
        var item = poll();
        if (item == null) {
            return null;
        }

        if (item.getFirstTsMillis() >= decimBeforeTsMillis) {
            return item;
        }

        if (item.getLastTsMillis() < decimFromTsMillis) {
            return item;
        }

        last = item;
        return pollDecimRange();
    }

    private Item poll() {
        if (last != null) {
            var result = last;
            last = null;
            return result;
        }

        return it.next();
    }

    private Item pollDecimRange() {
        List<AggrGraphDataListIterator> iterators = new ArrayList<>();
        long firstTsMillis = last.getFirstTsMillis();
        long lastTsMillis;
        int elapsedBytes = 0;
        do {
            iterators.add(last.iterator());
            lastTsMillis = last.getLastTsMillis();
            elapsedBytes += last.getElapsedBytes();
            last = it.next();
        } while (last != null && last.getFirstTsMillis() < decimBeforeTsMillis);

        var decimIt = DecimatingAggrGraphDataIterator.of(it.type(), ConcatAggrGraphDataIterator.of(iterators), policy, now);
        return new ItemIterator(decimIt, firstTsMillis, lastTsMillis, elapsedBytes);
    }
}
