package ru.yandex.stockpile.server.shard.merge;

import java.time.Instant;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.BinaryAggrGraphDataListIterator;
import ru.yandex.solomon.codec.bits.BitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class ItemFrame implements Item {
    private final StockpileFormat format;
    private final MetricType type;
    private final int mask;
    private final int record;
    private final long firstTsMillis;
    private final long lastTsMillis;
    private final BitBuf buffer;

    public ItemFrame(
        StockpileFormat format,
        MetricType type,
        int mask,
        int record,
        long firstTsMillis,
        long lastTsMillis,
        BitBuf buffer)
    {
        this.format = format;
        this.type = type;
        this.mask = mask;
        this.record = record;
        this.firstTsMillis = firstTsMillis;
        this.lastTsMillis = lastTsMillis;
        this.buffer = buffer;
    }

    public StockpileFormat getFormat() {
        return format;
    }

    public MetricType getType() {
        return type;
    }

    public int getMask() {
        return mask;
    }

    public int getRecord() {
        return record;
    }

    public BitBuf getBuffer() {
        return buffer.asReadOnly();
    }

    @Override
    public long getFirstTsMillis() {
        return firstTsMillis;
    }

    @Override
    public long getLastTsMillis() {
        return lastTsMillis;
    }

    public int getElapsedBytes() {
        return buffer.bytesSize();
    }

    @Override
    public AggrGraphDataListIterator iterator() {
        return new BinaryAggrGraphDataListIterator(type, mask, buffer.asReadOnly(), record);
    }

    @Override
    public void visit(ItemVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public String toString() {
        return "ItemFrame{" +
            "mask=" + StockpileColumnSet.toString(mask) +
            ", record=" + DataSize.shortString(record) +
            ", firstTsMillis=" + Instant.ofEpochMilli(firstTsMillis) +
            ", lastTsMillis=" + Instant.ofEpochMilli(lastTsMillis) +
            ", bytes=" + DataSize.shortString(buffer.bytesSize()) +
            '}';
    }
}
