package ru.yandex.stockpile.server.shard.merge;

import java.time.Instant;

import javax.annotation.Nullable;

import ru.yandex.solomon.codec.bits.BitBufAllocator;
import ru.yandex.solomon.codec.bits.ReadOnlyHeapBitBuf;
import ru.yandex.solomon.codec.compress.CompressStreamFactory;
import ru.yandex.solomon.codec.compress.TimeSeriesOutputStream;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
public class SplitCollector {
    private final StockpileFormat format;
    private final MetricType type;
    private final int mask;
    @Nullable
    private TimeSeriesOutputStream out;
    private long lastTsMillis;

    public SplitCollector(StockpileFormat format, MetricType type, int mask) {
        this.format = format;
        this.type = type;
        this.mask = mask;
    }

    public TimeSeriesOutputStream getOrCreateOut() {
        if (out == null) {
            out = CompressStreamFactory.createOutputStream(type, mask);
        }

        return out;
    }

    public static Result collect(int elapsedBytes, long tsMillis, Iterator it) {
        return collect(StockpileFormat.CURRENT, it.type(), it.columnSetMask(), elapsedBytes, tsMillis, it);
    }

    public static Result collect(StockpileFormat format, MetricType type, int mask, int elapsedBytes, long tsMillis, Iterator it) {
        var first = new SplitCollector(format, type, mask);
        var second = new CompressCollector(format, type, mask, BitBufAllocator.buffer(elapsedBytes));

        Item item;
        long lastTsMillis = 0;
        while ((item = it.next()) != null) {
            if (item.getFirstTsMillis() <= lastTsMillis) {
                lastTsMillis = item.getLastTsMillis();
                throw new IllegalStateException("First ts at frame  " + item + " <= then last appended " + Instant.ofEpochMilli(lastTsMillis));
            }

            if (item.getFirstTsMillis() >= tsMillis) {
                item.visit(second);
                continue;
            }

            if (item.getLastTsMillis() < tsMillis) {
                first.appendAll(item);
                continue;
            }

            first.appendPart(tsMillis, item, second);
        }

        return new Result(first.finish(), second.finish());
    }

    private void appendAll(Item item) {
        var it = item.iterator();
        var output = getOrCreateOut();
        output.ensureCapacity(item.getElapsedBytes());
        var point = RecyclableAggrPoint.newInstance();
        try {
            while (it.next(point)) {
                output.writePoint(mask, point);
                lastTsMillis = point.tsMillis;
            }
        } finally {
            point.recycle();
        }
    }

    private void appendPart(long beforeTsMillis, Item item, CompressCollector second) {
        var it = item.iterator();
        var output = getOrCreateOut();
        output.ensureCapacity(item.getElapsedBytes());

        var point = RecyclableAggrPoint.newInstance();
        try {
            while (it.next(point)) {
                if (point.tsMillis < beforeTsMillis) {
                    output.writePoint(mask, point);
                    lastTsMillis = point.tsMillis;
                } else {
                    second.appendPoint(point);
                    second.appendIterator(it, item.getElapsedBytes());
                }
            }
        } finally {
            point.recycle();
        }
    }

    private CompressResult finish() {
        if (out == null) {
            return new CompressResult(format, type, mask, ReadOnlyHeapBitBuf.EMPTY, 0, 0, 0);
        }

        out.closeFrame();
        return new CompressResult(format, type, mask, out.getCompressedData(), out.recordCount(), lastTsMillis, 1);
    }

    public static class Result {
        public final CompressResult before;
        public final CompressResult after;

        private Result(CompressResult before, CompressResult after) {
            this.before = before;
            this.after = after;
        }
    }
}
