package ru.yandex.stockpile.server.shard.stat;

import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.memory.layout.MemMeasurableSubsystem;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.server.shard.StockpileMetricReadResponse;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class GlobalUsageStats implements MemMeasurableSubsystem {
    public final UsageStatsCollector stats = new UsageStatsCollector();

    public AggrGraphDataIterable lazyRead(RequestProducer producer, MetricHeader header, AggrGraphDataIterable source) {
        return new AggrGraphDataIterable() {
            private boolean estimated;

            @Override
            public int getRecordCount() {
                return source.getRecordCount();
            }

            @Override
            public int elapsedBytes() {
                return source.elapsedBytes();
            }

            @Override
            public AggrGraphDataListIterator iterator() {
                var it = source.iterator();
                if (estimated) {
                    return it;
                }

                return new AggrGraphDataListIterator(source.columnSetMask()) {
                    private int records;
                    @Override
                    public boolean next(AggrPoint target) {
                        if (it.next(target)) {
                            records++;
                            return true;
                        }

                        if (!estimated) {
                            estimated = true;
                            read(header.getOwnerProjectId(), header.getOwnerShardId(), producer, header.getType(), records);
                        }

                        return false;
                    }
                };
            }

            @Override
            public int columnSetMask() {
                return source.columnSetMask();
            }
        };
    }

    public void read(RequestProducer producer, StockpileMetricReadResponse response) {
        var h = response.getHeader();
        var records = response.getTimeseries().getRecordCount();
        stats.read(h.getOwnerProjectId(), h.getOwnerShardId(), producer, h.getType(), records);
    }

    public void read(int projectId, int ownerId, RequestProducer producer, MetricType type, int records) {
        stats.read(projectId, ownerId, producer, type, records);
    }

    public void read(EProjectId projectId, int ownerId, RequestProducer producer, MetricType type, int records) {
        stats.read(projectId.getNumber(), ownerId, producer, type, records);
    }

    public void write(EProjectId projectId, int ownerId, MetricType type, long records) {
        stats.write(projectId.getNumber(), ownerId, type, records);
    }

    public void write(int projectId, int ownerId, MetricType type, long records) {
        stats.write(projectId, ownerId, type, records);
    }

    public void write(MetricArchiveGeneric archive) {
        write(archive.getOwnerProjectId(), archive.getOwnerShardId(), archive.getType(), archive.getRecordCount());
    }

    public void write(Collection<MetricArchiveMutable> archives) {
        stats.write(archives);
    }

    public void write(TxWriteStats summary) {
        stats.write(summary);
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addMemory(GlobalUsageStats.class.getSimpleName(), stats.memorySizeIncludingSelf());
    }
}
