package ru.yandex.stockpile.server.shard.stat;

import java.util.EnumMap;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;
import ru.yandex.stockpile.server.shard.LogOrSnapshot;
import ru.yandex.stockpile.server.shard.ProcessType;
import ru.yandex.stockpile.server.shard.ReadResultStatus;

/**
 * @author Stepan Koltsov
 */
@Component
@ParametersAreNonnullByDefault
public class StockpileShardAggregatedStats {

    private final EnumMap<ProcessType, Rate> processStarted;

    public final EnumMap<ReadResultStatus, Rate> readProcessingStatus;

    public final EnumMap<LogOrSnapshot, Histogram> writeElapsedTimeByLogType;
    public final Histogram writeLogTimeHistogramByRecords;

    /**
     * time which write request spend in write queue
     */
    public final Histogram writeInQueueTimeHistogram;
    public final Histogram readInQueueTimeHistogram;
    public final Histogram allocateLocalIdsInQueueTimeHistogram;

    /**
     * time spend in preparing write transaction from batch of write requests
     */
    public final Histogram writeTxPrepareTimeHistogram;

    /**
     * time spend in cache update
     */
    public final Histogram writeTxUpdateCacheTimeHistogram;

    /**
     * time spend in mem state update
     */
    public final Histogram writeTxUpdateMemStateTimeHistogram;

    public final Histogram readAmplification;
    public final Histogram readMetricFromDiskElapsedTimeMillis;

    public final Rate writeLogOpsStarted;
    public final Rate writeLogOpsCompleted;
    public final Histogram writeLogByteSize;
    public final Rate readMetricsRate;
    public final Rate readPointsRate;

    public final Meter readUtime;

    public StockpileShardAggregatedStats(MetricRegistry registry) {
        writeLogOpsStarted = registry.rate("stockpile.write.tx.ops.started");
        writeLogOpsCompleted = registry.rate("stockpile.write.tx.ops.completed");
        writeLogByteSize = registry.histogramRate("stockpile.write.tx.bytesSize", Histograms.exponential(25, 2, 32));
        writeTxPrepareTimeHistogram = registry.histogramRate("stockpile.write.tx.prepare.elapsedTimeMs", Histograms.exponential(17, 2, 1));
        writeTxUpdateCacheTimeHistogram = registry.histogramRate("stockpile.write.tx.updateCache.elapsedTimeMs", Histograms.exponential(17, 2, 1));
        writeTxUpdateMemStateTimeHistogram = registry.histogramRate("stockpile.write.tx.updateMemState.elapsedTimeMs", Histograms.exponential(17, 2, 1));
        writeInQueueTimeHistogram = registry.histogramRate("stockpile.write.requests.inQueue.elapsedTimeMs", Histograms.exponential(15, 2, 16));
        readInQueueTimeHistogram = registry.histogramRate("stockpile.read.requests.inQueue.elapsedTimeMs", Histograms.exponential(13, 2, 16));
        allocateLocalIdsInQueueTimeHistogram = registry.histogramRate("stockpile.allocateIds.requests.inQueue.elapsedTimeMs", Histograms.exponential(13, 2, 16));
        writeLogTimeHistogramByRecords = registry.histogramRate("stockpile.write.log.records.elapsedTimeMs", Histograms.exponential(13, 2, 16));
        readAmplification = registry.histogramRate("stockpile.read.amplification", Histograms.exponential(9, 2));
        readMetricFromDiskElapsedTimeMillis = registry.histogramRate("stockpile.read.disk.elapsedTimeMillis",
            Histograms.exponential(13, 2, 16));

        writeElapsedTimeByLogType = new EnumMap<>(LogOrSnapshot.class);
        for (LogOrSnapshot value : LogOrSnapshot.values()) {
            Histogram histogram = registry.histogramRate("stockpile.write.log.elapsedTimeMs", Labels.of("type", value.name()), Histograms.exponential(13, 2, 16));
            writeElapsedTimeByLogType.put(value, histogram);
        }

        readProcessingStatus = new EnumMap<>(ReadResultStatus.class);
        for (ReadResultStatus status : ReadResultStatus.values()) {
            readProcessingStatus.put(status, registry.rate("stockpile.read.sensors.status", Labels.of("status", status.name())));
        }

        processStarted = EnumMetrics.rates(ProcessType.class, registry, "stockpile.process.started", "type");
        readMetricsRate = registry.rate("stockpile.read.sensors.rate");
        readPointsRate = registry.rate("stockpile.read.points.rate");
        readUtime = registry.fifteenMinutesMeter("stockpile.read.utimeNanos");
    }

    public void processStarted(ProcessType processType) {
        processStarted.get(processType).inc();
    }

    public void readCompleted(ReadResultStatus status) {
        readProcessingStatus.get(status).inc();
    }
}
