package ru.yandex.stockpile.server.shard.stat;

import java.util.concurrent.TimeUnit;

import it.unimi.dsi.fastutil.longs.Long2LongOpenHashMap;

import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
public class UsageStats implements MetricSupplier, MemMeasurable {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(UsageStats.class) +
            (MemoryCounter.objectSelfSizeLayout(Rate.class) * 4) + MemoryCounter.objectSelfSizeLayout(Histogram.class);

    private final Rate writeRecords = new Rate();
    private final Rate writeMetrics = new Rate();
    private final Rate readRecords = new Rate();
    private final Rate readMetrics = new Rate();
    private final Histogram lagMetric = Histogram.newRate(Histograms.exponential(11, 2, 1));
    private final boolean needReportLag;

    public UsageStats(boolean needReportLag) {
        this.needReportLag = needReportLag;
    }

    @Override
    public int estimateCount() {
        return 4;
    }

    @Override
    public void append(long tsMillis, Labels common, MetricConsumer c) {
        append(tsMillis, common, c, "stockpile.write.records.rate", writeRecords);
        append(tsMillis, common, c, "stockpile.write.sensors.rate", writeMetrics);
        if (writeMetrics.get() > 0 && needReportLag) {
            baseAppend(tsMillis, common, c, "stockpile.write.sensors.lag_seconds", lagMetric);
        }
        append(tsMillis, common, c, "stockpile.read.records.rate", readRecords);
        append(tsMillis, common, c, "stockpile.read.sensors.rate", readMetrics);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE;
    }

    public void read(long records) {
        readMetrics.inc();
        readRecords.add(records);
    }

    public void write(long records) {
        writeMetrics.inc();
        writeRecords.add(records);
    }

    public void write(TxWriteStats.UsageMetric metrics, long currentTs) {
        writeMetrics.add(metrics.getMetricsCount());
        writeRecords.add(metrics.getPoints());
        writeLag(metrics.getLastTs2Count(), currentTs);
    }

    public void writeLag(Long2LongOpenHashMap lastTs2Count, long currentTs) {
        var it = lastTs2Count.long2LongEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            lagMetric.record(TimeUnit.MILLISECONDS.toSeconds(currentTs - entry.getLongKey()),
                    entry.getLongValue());
        }
    }

    public void combine(UsageStats stats) {
        writeRecords.combine(stats.writeRecords);
        writeMetrics.combine(stats.writeMetrics);
        readRecords.combine(stats.readRecords);
        readMetrics.combine(stats.readMetrics);
        lagMetric.combine(stats.lagMetric);
    }

    private void append(long tsMillis, Labels common, MetricConsumer c, String name, Rate rate) {
        if (rate.get() == 0) {
            return;
        }
        baseAppend(tsMillis, common, c, name, rate);
    }

    private void baseAppend(long tsMillis, Labels common, MetricConsumer c, String name, Metric histogram) {
        c.onMetricBegin(histogram.type());
        {
            c.onLabelsBegin(common.size() + 1);
            {
                common.forEach(c::onLabel);
                c.onLabel("sensor", name);
            }
            c.onLabelsEnd();
        }
        histogram.accept(tsMillis, c);
        c.onMetricEnd();
    }
}
