package ru.yandex.stockpile.server.shard.stat;

import java.util.Collection;
import java.util.concurrent.locks.StampedLock;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Vladimir Gordiychuk
 */
@SuppressWarnings("ProtocolBufferOrdinal")
public class UsageStatsCollector implements MetricSupplier, MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(UsageStatsCollector.class);
    private static final String[] PROJECTS =
        Stream.concat(
            Stream.of(EProjectId.values()).map(EProjectId::name),
            Stream.of("total")
        ).toArray(String[]::new);
    private static final String[] TYPES =
        Stream.concat(
            Stream.of(MetricType.values()).map(MetricType::name),
            Stream.of("total")
        ).toArray(String[]::new);
    private static final String[] PRODUCER =
        Stream.concat(
            Stream.of(RequestProducer.values()).map(Enum::name),
            Stream.of("total")
        ).toArray(String[]::new);
    private static final int PROJECTS_TOTAL = PROJECTS.length - 1;
    private static final int OWNER_TOTAL = 0;
    private static final int KIND_TOTAL = TYPES.length - 1;
    private static final int PRODUCER_TOTAL = PRODUCER.length - 1;

    private Long2ObjectOpenHashMap<UsageStats> byKey = new Long2ObjectOpenHashMap<>();
    private StampedLock sl = new StampedLock();

    @Override
    public int estimateCount() {
        return 4 * (byKey.size() + 1);
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        var total = new Long2ObjectOpenHashMap<UsageStats>();
        var copy = copyMap();
        var it = copy.long2ObjectEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            var key = entry.getLongKey();
            var value = entry.getValue();

            var projectId = UsageKey.project(key);
            var typeId = UsageKey.type(key);
            var ownerId = UsageKey.shardId(key);
            var producer = UsageKey.producer(key);

            if (ownerId != OWNER_TOTAL) {
                // TODO: rename kind -> type
                Labels labels = Labels.of(
                    "projectId", PROJECTS[projectId],
                    "kind", TYPES[typeId],
                    "producer", PRODUCER[producer],
                    "ownerId", Integer.toUnsignedString(ownerId));
                value.append(tsMillis, commonLabels.addAll(labels), consumer);
            }

            addAggregates(total, projectId, producer, typeId, ownerId, value);
        }

        appendAggregates(tsMillis, commonLabels, consumer, total);
    }

    private Long2ObjectOpenHashMap<UsageStats> copyMap() {
        long stamp = sl.readLock();
        try {
            return new Long2ObjectOpenHashMap<>(byKey);
        } finally {
            sl.unlockRead(stamp);
        }
    }

    private void appendAggregates(long tsMillis, Labels common, MetricConsumer consumer, Long2ObjectOpenHashMap<UsageStats> total) {
        var it = total.long2ObjectEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            var key = entry.getLongKey();
            var value = entry.getValue();

            var projectId = UsageKey.project(key);
            var typeId = UsageKey.type(key);
            var ownerId = UsageKey.shardId(key);
            var producer = UsageKey.producer(key);

            Labels labels = Labels.of(
                "projectId", PROJECTS[projectId],
                "kind", TYPES[typeId],
                "producer", PRODUCER[producer],
                "ownerId", ownerId == OWNER_TOTAL ? "total" : Integer.toUnsignedString(ownerId));
            value.append(tsMillis, common.addAll(labels), consumer);
        }
    }

    private void addAggregates(Long2ObjectOpenHashMap<UsageStats> total, int projectId, int producer, int kindId, int ownerId, UsageStats value) {
        combine(total, UsageKey.key(PROJECTS_TOTAL, producer, kindId, OWNER_TOTAL), value, false);
        combine(total, UsageKey.key(PROJECTS_TOTAL, producer, KIND_TOTAL, OWNER_TOTAL), value, false);
        combine(total, UsageKey.key(PROJECTS_TOTAL, PRODUCER_TOTAL, KIND_TOTAL, OWNER_TOTAL), value, true);

        combine(total, UsageKey.key(projectId, PRODUCER_TOTAL, kindId, ownerId), value, false);
        combine(total, UsageKey.key(projectId, PRODUCER_TOTAL, KIND_TOTAL, ownerId), value, true);
        combine(total, UsageKey.key(projectId, PRODUCER_TOTAL, KIND_TOTAL, OWNER_TOTAL), value, true);

        combine(total, UsageKey.key(projectId, producer, KIND_TOTAL, ownerId), value, false);
        combine(total, UsageKey.key(projectId, producer, KIND_TOTAL, OWNER_TOTAL), value, false);

        combine(total, UsageKey.key(projectId, producer, kindId, OWNER_TOTAL), value, false);
    }

    private void combine(Long2ObjectOpenHashMap<UsageStats> map, long key, UsageStats stats, boolean needReportLag) {
        var target = map.get(key);
        if (target == null) {
            target = new UsageStats(needReportLag);
            map.put(key, target);
        }

        target.combine(stats);
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += UsageStats.SELF_SIZE * (byKey.size() + 1);
        return size;
    }

    public void write(int projectId, int ownerId, MetricType type, long records) {
        long key = UsageKey.key(projectId, type.ordinal(), ownerId);
        stats(key).write(records);
    }

    public void write(Collection<MetricArchiveMutable> archives) {
        write(TxWriteStats.of(archives));
    }

    public void write(TxWriteStats update) {
        var summary = update.summary;

        // fast path
        long currentTs = System.currentTimeMillis();
        var it = summary.long2ObjectEntrySet().fastIterator();
        long stamp = sl.tryOptimisticRead();
        if (stamp != 0) {
            while (it.hasNext()) {
                var entry = it.next();
                TxWriteStats.UsageMetric metrics = entry.getValue();

                UsageStats stats = byKey.get(entry.getLongKey());
                if (sl.validate(stamp) && stats != null) {
                    stats.write(metrics, currentTs);
                } else {
                    stats(entry.getLongKey()).write(metrics, currentTs);
                    break;
                }
            }
        }
        // slow path
        while (it.hasNext()) {
            var entry = it.next();
            TxWriteStats.UsageMetric metrics = entry.getValue();
            stats(entry.getLongKey()).write(metrics, currentTs);
        }
    }

    public void read(int projectId, int ownerId, RequestProducer producer, MetricType type, long records) {
        long key = UsageKey.key(projectId, producer.ordinal(), type.ordinal(), ownerId);
        stats(key).read(records);
    }

    private UsageStats stats(long key) {
        var result = get(key);
        if (result != null) {
            return result;
        }

        return setIfAbsent(key, new UsageStats(false));
    }

    @Nullable
    private UsageStats get(long key) {
        long stamp = sl.tryOptimisticRead();
        if (stamp != 0) {
            var result = byKey.get(key);
            if (sl.validate(stamp)) {
                return result;
            }
        }

        stamp = sl.readLock();
        try {
            return byKey.get(key);
        } finally {
            sl.unlockRead(stamp);
        }
    }

    private UsageStats setIfAbsent(long key, UsageStats value) {
        long stamp = sl.writeLock();
        try {
            var prev = byKey.get(key);
            if (prev != null) {
                return prev;
            }

            byKey.put(key, value);
            return value;
        } finally {
            sl.unlockWrite(stamp);
        }
    }
}
