package ru.yandex.stockpile.server.www;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.RedirectException;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlProgressBar;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.shard.MergeKind;
import ru.yandex.stockpile.server.shard.SnapshotTs;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Import({
    ManagerWriterContext.class,
    StockpileLocalShards.class
})
public class StockpileLocalShardsSnapshotWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private StockpileLocalShards shards;

    private static String formatDuration(long millis) {
        boolean isNegative = millis < 0;
        millis = Math.abs(millis);
        for (TimeUnit unit : Arrays.asList(TimeUnit.DAYS, TimeUnit.HOURS, TimeUnit.MINUTES, TimeUnit.SECONDS)) {
            long mod = unit.toMillis(1);
            if (millis > mod) {
                millis -= millis % mod;
                break;
            }
        }

        String result = DurationUtils.formatDurationMillis(millis);
        if (isNegative) {
            result = "-" + result;
        }
        return result;
    }

    @Bean
    public RootLink stockpileLocalMergeLinks() {
        return new RootLink("/snapshot", "Stockpile snapshots");
    }

    @RequestMapping(value = "/snapshot", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> stockpileLocalShards(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> stockpileLocalShardsImpl(sortBy));
    }

    private String stockpileLocalShardsImpl(int sortBy) {
        List<Record> records = this.shards.stream()
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.shardId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Stockpile snapshots", contect, columns, records, sortBy).genString();
    }

    @RequestMapping("/snapshot/schedule")
    public CompletableFuture<String> makeSnapshot(
        @RequestParam(value = "shardId") int shardId,
        @RequestParam(value = "level") SnapshotLevel level,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> makeSnapshotImpl(shardId, level, request));
    }

    private String makeSnapshotImpl(int shardId, SnapshotLevel level, ServerHttpRequest request) {
        var shard = shards.getShardById(shardId);
        if (shard == null) {
            return redirectBack(request);
        }

        switch (level) {
            case DAILY:
                shard.forceMerge(MergeKind.DAILY);
                break;
            case ETERNITY:
                shard.forceMerge(MergeKind.ETERNITY);
                break;
            case TWO_HOURS:
                shard.forceSnapshot();
                break;
        }

        return redirectBack(request);
    }

    private String redirectBack(ServerHttpRequest request) {
        String referer = request.getHeaders().getFirst("Referer");
        if (Strings.isNullOrEmpty(referer)) {
            throw new RedirectException("/snapshot");
        }

        throw new RedirectException(referer);
    }

    private List<Column<Record>> makeColumns() {
        List<Column<Record>> columns = new ArrayList<>();
        columns.add(
            Column.of(
                "ShardId",
                r -> {
                    String shardId = Integer.toString(r.shardId);
                    return new AHref(namedObjectLink(new NamedObjectId(StockpileShard.class, shardId)), shardId);
                },
                Comparator.comparingInt(o -> o.shardId)));

        columns.add(Column.of("State", r -> r.state, Comparator.comparing(o2 -> o2.state)));

        columns.add(
            Column.of("Uptime",
                r -> formatDuration(r.uptimeMillis),
                Comparator.comparingLong(o2 -> o2.uptimeMillis)
            )
        );

        for (SnapshotLevel level : SnapshotLevel.values()) {
            columns.add(
                Column.of(level.shortName + " last time",
                    r -> {
                        long millis = r.lastMillis.get(level);
                        final String value;
                        if (millis == SnapshotTs.SpecialTs.NEVER.value) {
                            value = "NEVER";
                        } else if (millis == SnapshotTs.SpecialTs.NOT_READY.value) {
                            value = "NOT_READY";
                        } else {
                            value = formatDuration(millis);
                        }

                        return (WritableToHtml) mw -> {
                            var hw = mw.getHtmlWriter();
                            hw.write(value);
                            hw.write(" ");
                            hw.aHref("/snapshot/schedule?shardId=" + r.shardId + "&level=" + level.name(), () -> {
                                hw.tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-repeat"));
                            });
                        };
                    },
                    Comparator.comparingLong(o2 -> o2.lastMillis.get(level))
                )
            );
        }

        for (SnapshotLevel level : SnapshotLevel.values()) {
            columns.add(
                Column.of(level.shortName + " progress",
                    r -> {
                        double progress = r.progress.get(level);
                        if (progress == 100.0) {
                            return "";
                        } else {
                            return new HtmlProgressBar(progress);
                        }
                    },
                    Comparator.comparingDouble(o2 -> o2.progress.get(level))
                )
            );
        }

        return columns;
    }

    private static class Record implements TableRecord {
        int shardId;
        StockpileShard.LoadState state;
        long uptimeMillis;
        EnumMap<SnapshotLevel, Long> lastMillis = new EnumMap<>(SnapshotLevel.class);
        EnumMap<SnapshotLevel, Double> progress = new EnumMap<>(SnapshotLevel.class);

        public static Record of(StockpileShard shard) {
            long now = System.currentTimeMillis();
            var r = new Record();
            r.shardId = shard.shardId;
            r.uptimeMillis = now - shard.stockpileShardCreatedInstantMillis;
            r.state = shard.getLoadState();
            for (SnapshotLevel level : SnapshotLevel.values()) {
                var time = shard.latestSnapshotTime(level).getTsOrSpecial();
                if (time <= 0) {
                    r.lastMillis.put(level, time);
                } else {
                    r.lastMillis.put(level, now - time);
                }

                r.progress.put(level, shard.getSnapshotProgress(level));
            }

            return r;
        }
    }
}
