package ru.yandex.stockpile.server.www;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlProgressBar;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Sergey Polovko
 */
@RestController
@Import({
    ManagerWriterContext.class,
    StockpileLocalShards.class
})
public class StockpileLocalShardsWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private StockpileLocalShards shards;

    @Bean
    public RootLink stockpileLocalShardLinks() {
        return new RootLink("/local-shards", "Stockpile local shards");
    }

    @RequestMapping(value = {"/stockpile-local-shards", "/local-shards"}, produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> stockpileLocalShards(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> stockpileLocalShardsImpl(sortBy));
    }

    private String stockpileLocalShardsImpl(int sortBy) {
        List<Record> records = this.shards.stream()
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.shardId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Stockpile local shards", contect, columns, records, sortBy).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
            Column.of(
                "ShardId",
                r -> {
                    String shardId = Integer.toString(r.shardId);
                    return new AHref(namedObjectLink(new NamedObjectId(StockpileShard.class, shardId)), shardId);
                },
                Comparator.comparingInt(o -> o.shardId)),

            Column.of("TabletId", r -> Long.toUnsignedString(r.tabletId), Comparator.comparingLong(r -> r.tabletId)),
            Column.of("State", r -> r.state, Comparator.comparing(o2 -> o2.state)),
            Column.of(
                "Load",
                r -> {
                    if (r.state != StockpileShard.LoadState.DONE) {
                        return new HtmlProgressBar(r.loadProgress);
                    }

                    return DurationUtils.formatDurationMillis(r.loadMillis);
                },
                Comparator.comparingLong(o2 -> o2.loadMillis)),
            Column.of("Uptime",
                r -> DurationUtils.formatDurationMillis(r.uptimeMillis),
                Comparator.comparingLong(o2 -> o2.uptimeMillis)
            ),
            Column.of("Cpu",
                r -> String.format("%.5f", r.cpu),
                Comparator.comparingDouble(o2 -> o2.cpu)
            ),
            Column.of("Memory",
                r -> DataSize.shortString(r.memory)+"B",
                Comparator.comparingLong(o2 -> o2.memory)
            ),
            Column.of("Read Metrics/sec",
                r -> DataSize.shortString(Math.round(r.readMetricsRps)),
                Comparator.comparingDouble(o2 -> o2.readMetricsRps)
            ),
            Column.of("Write Metrics/sec",
                r -> DataSize.shortString(Math.round(r.writeRecordsRps)),
                Comparator.comparingDouble(o2 -> o2.writeRecordsRps)
            ),
            Column.of("Errors",
                r -> DataSize.shortString(r.errors),
                Comparator.comparingLong(o2 -> o2.errors)
            ),
            Column.of("Last Error",
                r -> {
                  if (r.lastErrorInstant == 0) {
                      return "none";
                  } else {
                      return Instant.ofEpochMilli(r.lastErrorInstant);
                  }
                },
                Comparator.comparingLong(o2 -> o2.lastErrorInstant)
            ),
            Column.of("Kick", r -> {
                return (WritableToHtml) mw -> {
                    var hw = mw.getHtmlWriter();
                    hw.aHref("/balancer/kickShard?shardId=" + r.shardId, () -> {
                        hw.tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-transfer"));
                    });
                };
            }, Comparator.comparingInt(r -> r.shardId))
        );
    }

    private static class Record implements TableRecord {
        int shardId;
        long tabletId;
        StockpileShard.LoadState state;
        long loadMillis;
        double loadProgress;
        long uptimeMillis;
        double cpu;
        long memory;
        double readMetricsRps;
        double writeRecordsRps;
        long errors;
        long lastErrorInstant;

        public static Record of(StockpileShard shard) {
            var r = new Record();
            r.shardId = shard.shardId;
            r.tabletId = shard.kvTabletId;
            r.state = shard.getLoadState();
            r.uptimeMillis = System.currentTimeMillis() - shard.stockpileShardCreatedInstantMillis;
            r.loadMillis = shard.getStats().loadDurationMillis;
            r.loadProgress = shard.getLoadingProgress();
            r.cpu = shard.metrics().utimeNanos.getRate(TimeUnit.SECONDS) * 1e-6;
            r.memory = shard.memorySizeIncludingSelf();
            r.readMetricsRps = shard.metrics.read.avgMetricsRps.getRate(TimeUnit.SECONDS);
            r.writeRecordsRps = shard.metrics.write.avgRecordsRps.getRate(TimeUnit.SECONDS);
            r.errors = shard.metrics.errors.get();
            r.lastErrorInstant = shard.lastErrorInstant;
            return r;
        }
    }
}
