package ru.yandex.stockpile.server.www;

import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.selfmon.mon.HttpMetricRegistryEncoder;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsLevel;
import ru.yandex.stockpile.server.shard.StockpileExecutor;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileScheduledExecutor;
import ru.yandex.stockpile.server.shard.stat.GlobalUsageStats;
import ru.yandex.stockpile.server.shard.stat.UsageStatsCollector;

/**
 * @author Vladimir Gordiychuk
 */
@Import(
    GlobalUsageStats.class
)
@Controller
public class StockpileResourceUsageController implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(StockpileResourceUsageController.class);
    private final StorageUsage storageUsage;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private volatile ResponseEntity<byte[]> latestUsage =
            ResponseEntity.status(HttpStatus.NO_CONTENT)
                    .body(new byte[0]);
    private volatile boolean closed;
    private volatile ScheduledFuture<?> scheduled;

    @Autowired
    public StockpileResourceUsageController(
            StockpileLocalShards shards,
            GlobalUsageStats usage,
            @StockpileExecutor ExecutorService executor,
            @StockpileScheduledExecutor ScheduledExecutorService timer)
    {
        this.storageUsage = new StorageUsage(shards, usage.stats);
        this.executor = executor;
        this.timer = timer;
        this.scheduled = timer.schedule(this::runScheduledRefresh, 5, TimeUnit.MILLISECONDS);
    }

    @Bean
    public RootLink stockpileResourceUsageLink() {
        return new RootLink("/resourceUsage/metrics", "Resource usage");
    }

    @RequestMapping("/resourceUsage/metrics")
    public ResponseEntity<byte[]> metrics(ServerHttpRequest request) {
        return latestUsage;
    }

    private void runScheduledRefresh() {
        if (closed) {
            return;
        }

        executor.execute(() -> {
            try {
                latestUsage = HttpMetricRegistryEncoder.encode(storageUsage, MetricFormat.SPACK, CompressionAlg.LZ4);
            } catch (Throwable e) {
                logger.error("Failed refresh alert status", e);
            }

            if (closed) {
                return;
            }

            scheduled = timer.schedule(this::runScheduledRefresh, 30, TimeUnit.SECONDS);
        });
    }

    @Override
    public void close() {
        closed = true;
        var copy = scheduled;
        if (copy != null) {
            copy.cancel(false);
        }
    }

    private static class StorageUsage implements MetricSupplier {
        private final StockpileLocalShards shards;
        private final UsageStatsCollector usage;
        private volatile int metricsCount;

        public StorageUsage(StockpileLocalShards shards, UsageStatsCollector usage) {
            this.shards = shards;
            this.usage = usage;
        }

        @Override
        public int estimateCount() {
            return metricsCount;
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            commonLabels = commonLabels.add("host", "const");
            int metricsCount = 0;
            var total = new IndexStatsLevel();
            for (var shard : shards) {
                var stats = shard.indexStats();
                total.combine(stats);
                metricsCount += stats.estimateCount();
            }
            total.append(tsMillis, commonLabels, consumer);
            metricsCount += total.estimateCount();
            usage.append(tsMillis, commonLabels, consumer);
            metricsCount += usage.estimateCount();
            this.metricsCount = metricsCount;
        }
    }
}
