package ru.yandex.stockpile.tool.server;

import java.util.Comparator;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;

import ru.yandex.grpc.utils.InternalGrpcService;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.internal.tool.StockpileToolsServiceGrpc;
import ru.yandex.stockpile.internal.tool.TForceLogSnapshotRequest;
import ru.yandex.stockpile.internal.tool.TForceLogSnapshotResponse;
import ru.yandex.stockpile.internal.tool.TStockpileToolServiceGrpc;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileToolServiceGrpc extends TStockpileToolServiceGrpc.TStockpileToolServiceImplBase implements InternalGrpcService {
    private StockpileLocalShards localShards;
    private ExecutorService executor;

    public StockpileToolServiceGrpc(StockpileLocalShards localShards, ExecutorService executor) {
        this.localShards = localShards;
        this.executor = executor;
    }

    @Override
    public void forceLogsSnapshot(TForceLogSnapshotRequest request, StreamObserver<TForceLogSnapshotResponse> responseObserver) {
        final long logSizeThreshold = request.getMinLogMegabytesSize() * (1 << 20);

        var shards = localShards.stream()
            .filter(shard -> logSize(shard) > logSizeThreshold)
            .sorted(Comparator.comparingLong(StockpileToolServiceGrpc::logSize))
            .collect(Collectors.toList());

        AtomicInteger cursor = new AtomicInteger();
        AsyncActorBody body = () -> {
            int pos = cursor.getAndIncrement();
            if (pos >= shards.size()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = shards.get(pos);
            return shard.forceSnapshot();
        };

        int inflight = request.getMaxInFlight() > 0 ? request.getMaxInFlight() : 5;
        AsyncActorRunner runner = new AsyncActorRunner(body, executor, inflight);
        runner.start()
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    responseObserver.onError(e);
                } else {
                    responseObserver.onNext(TForceLogSnapshotResponse.getDefaultInstance());
                    responseObserver.onCompleted();
                }
            });
    }

    private static long logSize(StockpileShard shard) {
        return shard.diskStats().get(FileKind.LOG).size();
    }

    // temporary proxy to migrate to a better named grpc service
    public static class Proxy extends StockpileToolsServiceGrpc.StockpileToolsServiceImplBase implements InternalGrpcService {

        private final StockpileToolServiceGrpc delegate;

        public Proxy(StockpileToolServiceGrpc delegate) {
            this.delegate = delegate;
        }

        @Override
        public void forceLogsSnapshot(TForceLogSnapshotRequest request, StreamObserver<TForceLogSnapshotResponse> responseObserver) {
            delegate.forceLogsSnapshot(request, responseObserver);
        }
    }
}
