package ru.yandex.stockpile.api.grpc;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.protobuf.ByteString;
import io.grpc.Server;
import io.grpc.inprocess.InProcessServerBuilder;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.UnpooledByteBufAllocator;
import io.netty.util.ResourceLeakDetector;
import io.netty.util.concurrent.DefaultThreadFactory;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.concurrency.limits.actors.LimiterNoop;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.grpc.utils.server.ServerMetrics;
import ru.yandex.grpc.utils.server.interceptors.MetricServerStreamTracer;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.Histogram;
import ru.yandex.solomon.model.protobuf.LogHistogram;
import ru.yandex.solomon.model.protobuf.LogHistogramConverter;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.SummaryDouble;
import ru.yandex.solomon.model.protobuf.SummaryInt64;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.LogsIndex;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.slog.ResolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.SnapshotLogDataBuilderImpl;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.api.CreateMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricDataResponse;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EColumnFlag;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TAllocateLocalIdsRequest;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TReadResponse;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.api.TWriteResponse;
import ru.yandex.stockpile.api.read.Converters;
import ru.yandex.stockpile.api.read.StockpileReadApi;
import ru.yandex.stockpile.api.read.StockpileReadApiMetrics;
import ru.yandex.stockpile.client.ColumnFlagMask;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileShardGlobals;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestContext;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.stockpile.client.TestUtil.metricId;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
    StockpileShardTestContext.class
})
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
public class StockpileGrpcServerTest {
    private static Logger logger = LoggerFactory.getLogger(StockpileGrpcServerTest.class);

    @Rule
    public TestName testName = new TestName();
    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withLookingForStuckThread(true)
            .withTimeout(1, TimeUnit.MINUTES)
            .build();

    @Autowired
    private KikimrKvClientInMem kikimrKvClientInMem;
    @Autowired
    private StockpileShardGlobals shardGlobals;
    private StockpileLocalShards shards;
    private Server server;
    private StockpileClient client;
    private ExecutorService grpcServerExecutorService;
    private ExecutorService grpcClientExecutorService;

    private static long timeToMillis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private static TPoint point(String time, double value) {
        return TPoint.newBuilder()
            .setTimestampsMillis(timeToMillis(time))
            .setDoubleValue(value)
            .build();
    }

    private static TPoint point(String time, double value, int count) {
        return TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis(time))
                .setDoubleValue(value)
                .setCount(count)
                .build();
    }

    @BeforeClass
    public static void beforeClass() {
        ResourceLeakDetector.setLevel(ResourceLeakDetector.Level.ADVANCED);
    }

    @Before
    public void setUp() throws Exception {
        logger.info("Run setUp for test: {}", testName.getMethodName());
        grpcServerExecutorService = Executors.newFixedThreadPool(2,
            new DefaultThreadFactory("stockpile-server-grpc"));
        grpcClientExecutorService = Executors.newFixedThreadPool(2,
            new DefaultThreadFactory("stockpile-client-grpc"));

        shards = new StockpileLocalShards();

        join(kikimrKvClientInMem.createKvTablets("/kv", 5));
        long[] tabletIds = join(kikimrKvClientInMem.resolveKvTablets("/kv"));

        for (int i = 0; i < tabletIds.length; i++) {
            final long tabletId = tabletIds[i];
            final int shardId = i + 1;

            StockpileShard shard = new StockpileShard(shardGlobals, shardId, tabletId, testName.getMethodName());
            shard.start();
            shard.waitForInitializedOrAnyError();
            shards.addShard(shard);
        }

        String serverName = UUID.randomUUID().toString();
        server = InProcessServerBuilder.forName(serverName)
            .addService(GrpcStockpileServiceFactory.create(
                    new StockpileReadApi(new StockpileReadApiMetrics()),
                    LimiterNoop.INSTANCE,
                    shards,
                    new MetricRegistry()))
            .addStreamTracerFactory(new MetricServerStreamTracer.Factory(new ServerMetrics()))
            .executor(grpcServerExecutorService)
            .build()
            .start();

        var clientOptions = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(15, TimeUnit.SECONDS)
                    .setChannelFactory(new InProcessChannelFactory())
                    .setRpcExecutor(grpcClientExecutorService))
            .setRetryStopStrategy(StopStrategies.neverStop())
            .setMetaDataRequestTimeOut(5, TimeUnit.SECONDS)
            .setExpireClusterMetadata(500, TimeUnit.MILLISECONDS)
            .setMetaDataRequestRetryDelay(30, TimeUnit.MILLISECONDS)
            .build();

        client = StockpileClients.createDynamic(
            List.of(serverName),
            clientOptions
        );

        while (client.getReadyShardsCount() < 5) {
            client.forceUpdateClusterMetaData().join();
        }
        logger.info("Run test: {}", testName.getMethodName());
    }

    @After
    public void tearDown() {
        logger.info("TearDown test: {}", testName.getMethodName());
        shards.forEach(StockpileShard::stop);
        if (server != null) {
            server.shutdownNow();
        }
        if (client != null) {
            client.close();
        }
        grpcClientExecutorService.shutdown();
        grpcServerExecutorService.shutdown();
    }

    @Test
    public void readOneEmptyDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-12T13:40:00Z", 1))
                .build());

        TReadRequest request = TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setCookie(123)
            .setFromMillis(System.currentTimeMillis())
            .build();

        TReadResponse response = client.readOne(request).join();

        TReadResponse expected = TReadResponse.newBuilder()
            .setStatus(EStockpileStatusCode.OK)
            .setCookie(123)
            .setMetricId(metricId)
            .setColumnMask(0)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .build();

        assertThat(response, equalTo(expected));
    }

    @Test
    public void readOneEmptyLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM);

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM)
                .setColumnMask(ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK)
                .addPoints(TPoint.newBuilder()
                    .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                    .setLogHistogram(LogHistogram.newBuilder()
                            .setZeroes(3)
                            .build())
                    .build())
                .build());

        TReadRequest request = TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setCookie(321)
            .setFromMillis(System.currentTimeMillis())
            .build();

        TReadResponse response = client.readOne(request).join();

        TReadResponse expected = TReadResponse.newBuilder()
            .setStatus(EStockpileStatusCode.OK)
            .setCookie(321)
            .setMetricId(metricId)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM)
            .setColumnMask(0)
            .build();

        assertThat(response, equalTo(expected));
    }

    @Test
    public void writeOneWithoutSeriesType() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        TWriteRequest request = TWriteRequest.newBuilder()
            .setMetricId(metricId) // missing TimeSeriesType
            .addPoints(point("2017-07-12T13:10:00Z", 1))
            .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.INVALID_REQUEST));
    }

    @Test
    public void writeOneDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        TWriteRequest request = TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-12T13:30:00Z", 3))
            .addPoints(point("2017-07-12T13:20:00Z", 2))
            .addPoints(point("2017-07-12T13:10:00Z", 1))
            .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void writeOneLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED);

        TWriteRequest request = TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK)
            .addPoints(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(3)
                    .setStartPower(1)
                    .addAllBuckets(Arrays.asList(2d, 1d, 5d))
                    .build())
                .build()
            ).build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void writeOneHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.HIST);

        TWriteRequest request = TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                .addPoints(TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(100d, 200d, 300d))
                                .addAllBuckets(Arrays.asList(5L, 10L, 0L))
                                .build())
                        .build())
                .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.OK));
    }

    @Test
    public void writeReadOneSummaryInt64() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.ISUMMARY);

        List<TPoint> source = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(42)
                                .setSum(1234)
                                .setMax(1000)
                                .setMin(2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:45:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(50)
                                .setSum(1240)
                                .setMax(1000)
                                .setMin(2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:50:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(80)
                                .setSum(2500)
                                .setMax(1200)
                                .setMin(-5)
                                .build())
                        .build());

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_ISUMMARY)
                .addAllPoints(source)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build());

        assertThat(response.getPointsList(), equalTo(source));
    }

    @Test
    public void writeReadOneSummaryDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DSUMMARY);

        List<TPoint> source = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(42)
                                .setSum(1234.3)
                                .setMax(1000.6)
                                .setMin(2.2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:45:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(50)
                                .setSum(1240.5)
                                .setMax(1000.6)
                                .setMin(2.2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:50:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(80)
                                .setSum(2500.1)
                                .setMax(1200.4)
                                .setMin(-5.1)
                                .build())
                        .build());

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DSUMMARY)
                .addAllPoints(source)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build());

        assertThat(response.getPointsList(), equalTo(source));
    }

    @Test
    public void readSummaryDoubleDefaultDownsampling() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DSUMMARY);

        List<TPoint> source = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(42)
                                .setSum(1234.3)
                                .setMax(1000.6)
                                .setMin(2.2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:45:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(50)
                                .setSum(1240.5)
                                .setMax(1000.6)
                                .setMin(2.2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:50:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(80)
                                .setSum(2500.1)
                                .setMax(1200.4)
                                .setMin(-5.1)
                                .build())
                        .build());

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DSUMMARY)
                .addAllPoints(source)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setFromMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setToMillis(timeToMillis("2017-07-12T14:00:00Z"))
                .setGridMillis(TimeUnit.MINUTES.toMillis(20L))
                .build());

        assertThat(response.getPointsList(), iterableWithSize(1));
        assertThat("Default downsampling for summary - sum", response.getPointsList().get(0), equalTo(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setSummaryDouble(SummaryDouble.newBuilder()
                        .setCount(172)
                        .setSum(4974.9)
                        .setMax(1200.4)
                        .setMin(-5.1)
                        .build())
                .setCount(3)
                .build()));
    }

    @Test
    public void readSummaryInt64DefaultDownsampling() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.ISUMMARY);

        List<TPoint> source = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(42)
                                .setSum(1234)
                                .setMax(1000)
                                .setMin(2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:45:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(50)
                                .setSum(1240)
                                .setMax(1000)
                                .setMin(2)
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:50:00Z"))
                        .setSummaryInt64(SummaryInt64.newBuilder()
                                .setCount(80)
                                .setSum(2500)
                                .setMax(1200)
                                .setMin(-5)
                                .build())
                        .build());

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_ISUMMARY)
                .addAllPoints(source)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setFromMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setToMillis(timeToMillis("2017-07-12T14:00:00Z"))
                .setGridMillis(TimeUnit.MINUTES.toMillis(20L))
                .build());

        assertThat(response.getPointsList(), iterableWithSize(1));
        assertThat("Default downsampling for summary - sum", response.getPointsList().get(0), equalTo(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setSummaryInt64(SummaryInt64.newBuilder()
                        .setCount(172)
                        .setSum(4974)
                        .setMax(1200)
                        .setMin(-5)
                        .build())
                .setCount(3)
                .build()));
    }

    @Test
    public void writeOneAndReadOneDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> expectPoints = Arrays.asList(
            point("2017-07-12T13:40:00Z", 1),
            point("2017-07-12T13:45:00Z", 2),
            point("2017-07-12T13:50:00Z", 3),
            point("2017-07-12T13:55:00Z", 4),
            point("2017-07-12T14:00:00Z", 5)
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addAllPoints(expectPoints)
            .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(response.getPointsList(), equalTo(expectPoints));
    }

    @Test
    public void writeOneAndReadOneLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED);

        List<TPoint> expectPoints = Arrays.asList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(2d, 1d, 0d, 0d, 5d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build(),

            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T10:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5)
                    .setStartPower(1)
                    .addAllBuckets(Arrays.asList(1d, 0d, 3d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build()
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK)
            .addAllPoints(expectPoints)
            .build()
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(response.getPointsList(), equalTo(expectPoints));
    }

    @Test
    public void writeOneReadOneHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.HIST);

        TPoint point = TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setHistogram(Histogram.newBuilder()
                        .addAllBounds(Arrays.asList(10d, 100d, 1000d))
                        .addAllBuckets(Arrays.asList(1L, 50L, 0L))
                        .build())
                .build();

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                .addPoints(point)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build());

        assertThat(response.getPointsList(), equalTo(Collections.singletonList(point)));
    }

    @Test
    public void writeReadManyHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.HIST);

        List<TPoint> points = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:15:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:30:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build()
        );

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                .addAllPoints(points)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build());

        assertThat(response.getPointsList(), equalTo(points));
    }

    @Test
    public void writeReadManyHistogramByTs() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.HIST);

        List<TPoint> points = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:15:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:30:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build()
        );

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                .addAllPoints(points)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .setFromMillis(timeToMillis("2017-07-13T05:10:00Z"))
                .setToMillis(timeToMillis("2017-07-13T05:20:00Z"))
                .build());

        assertThat(response.getPointsList(), equalTo(Collections.singletonList(points.get(1))));
    }

    @Test
    public void readOneFromTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T08:00:00Z", 4),
            point("2017-07-13T09:00:00Z", 5),
            point("2017-07-13T10:00:00Z", 6)
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setFromMillis(timeToMillis("2017-07-13T08:00:00Z"))
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void readOneToTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 1),
            point("2017-07-13T06:00:00Z", 2),
            point("2017-07-13T07:00:00Z", 3)
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
            .setToMillis(timeToMillis("2017-07-13T08:00:00Z")) // to exclusive
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void readOneRangeTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T06:00:00Z", 2),
            point("2017-07-13T07:00:00Z", 3),
            point("2017-07-13T08:00:00Z", 4)
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setFromMillis(timeToMillis("2017-07-13T06:00:00Z")) // inclusive
            .setToMillis(timeToMillis("2017-07-13T09:00:00Z")) // exclusive
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void readOneSortedByTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T10:00:00Z", 0))
            .addPoints(point("2017-07-13T08:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T05:00:00Z", 5))
            .addPoints(point("2017-07-13T09:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 4))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 5),
            point("2017-07-13T06:00:00Z", 4),
            point("2017-07-13T07:00:00Z", 3),
            point("2017-07-13T08:00:00Z", 2),
            point("2017-07-13T09:00:00Z", 1),
            point("2017-07-13T10:00:00Z", 0)
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void readOneDownSamplingDefaultDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:01:00Z", 5))
            .addPoints(point("2017-07-13T05:05:30Z", 4))
            .addPoints(point("2017-07-13T06:08:00Z", 2))
            .addPoints(point("2017-07-13T06:10:43Z", 3))
            .addPoints(point("2017-07-13T07:01:19Z", -2))
            .addPoints(point("2017-07-13T08:00:00Z", 9))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 4.5, 2),  // 5, 4
            point("2017-07-13T06:00:00Z", 2.5, 2),  // 2, 3
            point("2017-07-13T07:00:00Z", -2, 1), // -2
            point("2017-07-13T08:00:00Z", 9, 1)  // 9
        );

        // default aggregate function for double time series it's MAX
        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setGridMillis(TimeUnit.HOURS.toMillis(1L))
            .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
            .setToMillis(timeToMillis("2017-07-13T09:00:00Z"))
            .build());

        assertEquals(expectedPoints, response.getPointsList());
    }

    @Test
    public void readOneDownSamplingDefaultLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM);

        List<TPoint> source = Arrays.asList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(2d, 1d, 0d, 0d, 5d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build(),

            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:12:13Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(10)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(1d, 0d, 3d, 2d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build()
        );

        List<TPoint> expected = Collections.singletonList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5 + 10)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(2d + 1d, 1d + 0d, 0d + 3d, 0d + 2d, 5d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .setCount(2)
                .build()
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM)
            .setColumnMask(ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK)
            .addAllPoints(source)
            .build());

        // default aggregate function for log histogram time series it's SUM
        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setGridMillis(TimeUnit.HOURS.toMillis(1L))
            .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
            .setToMillis(timeToMillis("2017-07-13T06:00:00Z"))
            .build());

        assertThat(response.getPointsList(), equalTo(expected));
    }

    @Test
    public void readOneDownSamplingSumDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:01:00Z", 5))
            .addPoints(point("2017-07-13T05:05:30Z", 4))
            .addPoints(point("2017-07-13T06:08:00Z", 2))
            .addPoints(point("2017-07-13T06:10:43Z", 3))
            .addPoints(point("2017-07-13T07:01:19Z", -2))
            .addPoints(point("2017-07-13T08:00:00Z", 9))
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 5 + 4, 2),
            point("2017-07-13T06:00:00Z", 2 + 3, 2),
            point("2017-07-13T07:00:00Z", -2, 1),
            point("2017-07-13T08:00:00Z", 9, 1)
        );

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setAggregation(Aggregation.SUM)
            .setGridMillis(TimeUnit.HOURS.toMillis(1L))
            .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
            .setToMillis(timeToMillis("2017-07-13T09:00:00Z"))
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void writeOneNotAbleWithoutLocalId() {
        MetricId validmetricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        MetricId notValidmetricId = validmetricId.toBuilder()
            .clearLocalId()
            .build();

        TWriteRequest request = TWriteRequest.newBuilder()
            .setMetricId(notValidmetricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T06:08:00Z", 1))
            .build();

        TWriteResponse response = client.writeOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.INVALID_REQUEST));
    }

    @Test
    public void readOneNotAbleWithoutLocalId() {
        MetricId validmetricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        MetricId notValidmetricId = validmetricId.toBuilder()
            .clearLocalId()
            .build();

        TReadRequest request = TReadRequest.newBuilder()
            .setMetricId(notValidmetricId)
            .build();

        TReadResponse response = client.readOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.INVALID_REQUEST));
    }

    @Test
    public void deleteMetricNotAbleWithoutLocalId() {
        MetricId validmetricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        MetricId notValidmetricId = validmetricId.toBuilder()
            .clearLocalId()
            .build();

        DeleteMetricRequest request = DeleteMetricRequest.newBuilder()
            .setMetricId(notValidmetricId)
            .build();

        DeleteMetricResponse response = client.deleteMetric(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.INVALID_REQUEST));
    }

    @Test
    public void deleteMetricDataNotAbleWithoutLocalId() {
        MetricId validmetricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        MetricId notValidmetricId = validmetricId.toBuilder()
            .clearLocalId()
            .build();

        DeleteMetricDataRequest request = DeleteMetricDataRequest.newBuilder()
            .setMetricId(notValidmetricId)
            .setToMillis(Instant.now().toEpochMilli())
            .build();

        DeleteMetricDataResponse response = client.deleteMetricData(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.INVALID_REQUEST));
    }

    @Test
    public void deleteMetricDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:01:00Z", 5))
            .addPoints(point("2017-07-13T05:05:30Z", 4))
            .addPoints(point("2017-07-13T06:08:00Z", 2))
            .addPoints(point("2017-07-13T06:09:43Z", 3))
            .addPoints(point("2017-07-13T07:01:19Z", -2))
            .addPoints(point("2017-07-13T08:00:00Z", 9))
            .build());

        deleteMetricAndCheck(metricId);
    }

    @Test
    public void deleteMetricLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK)
            .addPoints(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(3)
                    .setStartPower(1)
                    .addAllBuckets(Arrays.asList(2d, 1d, 5d))
                    .build())
                .build()
            ).build());

        deleteMetricAndCheck(metricId);
    }

    @Test
    public void reusemetricIdAfterDelete() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T05:05:00Z", 2))
            .addPoints(point("2017-07-13T06:10:00Z", 3))
            .build());

        syncDeleteMetric(DeleteMetricRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        // We can't reuse localId until merge process not delete all data
        // Without it all next writes to this metric will be lost
        shards.forEach(StockpileShard::forceSnapshotSync);

        // TODO: restrict create metric until complete deletion?
        syncCreateMetric(CreateMetricRequest.newBuilder()
            .setMetricId(metricId)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .build());

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-18T10:00:00Z", 5),
            point("2017-07-18T10:05:00Z", 6),
            point("2017-07-18T10:10:00Z", 7)
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addAllPoints(expectedPoints)
            .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(response.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void deleteDataToTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        DeleteMetricDataRequest request = DeleteMetricDataRequest.newBuilder()
            .setMetricId(metricId)
            .setToMillis(timeToMillis("2017-07-13T08:00:00Z")) // exclusive
            .build();

        DeleteMetricDataResponse response = client.deleteMetricData(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T08:00:00Z", 4),
            point("2017-07-13T09:00:00Z", 5),
            point("2017-07-13T10:00:00Z", 6)
        );

        TReadResponse readResponse = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(readResponse.getPointsList(), equalTo(expectedPoints));
    }

    @Ignore("Delete from millis not supported yet")
    public void deleteDataFromTsDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        DeleteMetricDataRequest request = DeleteMetricDataRequest.newBuilder()
            .setMetricId(metricId)
            .setFromMillis(timeToMillis("2017-07-13T08:00:00Z")) // inclusive
            .build();

        DeleteMetricDataResponse response = client.deleteMetricData(request).join();
        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.OK));

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 1),
            point("2017-07-13T06:00:00Z", 2),
            point("2017-07-13T07:00:00Z", 3)
        );

        TReadResponse readResponse = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(readResponse.getPointsList(), equalTo(expectedPoints));
    }

    @Ignore("Delete range of data not supported yet")
    public void deleteDataFromRangeDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T08:00:00Z", 4))
            .addPoints(point("2017-07-13T09:00:00Z", 5))
            .addPoints(point("2017-07-13T10:00:00Z", 6))
            .build());

        DeleteMetricDataRequest request = DeleteMetricDataRequest.newBuilder()
            .setMetricId(metricId)
            .setFromMillis(timeToMillis("2017-07-13T06:00:00Z")) // inclusive
            .setToMillis(timeToMillis("2017-07-13T09:00:00Z")) // exclusive
            .build();

        DeleteMetricDataResponse response = client.deleteMetricData(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 1),
            point("2017-07-13T09:00:00Z", 5),
            point("2017-07-13T10:00:00Z", 6)
        );

        TReadResponse readResponse = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(readResponse.getPointsList(), equalTo(expectedPoints));
    }

    @Test
    public void writeCompressedOneNewFormatDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 5.3),
            point("2017-07-13T06:00:00Z", 4.1),
            point("2017-07-13T07:00:00Z", 3.3),
            point("2017-07-13T08:00:00Z", 2.9),
            point("2017-07-13T09:00:00Z", 1.0),
            point("2017-07-13T10:00:00Z", 0.0)
        );

        StockpileFormat format = StockpileFormat.byNumber(client.getCompatibleCompressFormat().upperEndpoint());
        checkWriteOneCompressed(metricId, expectedPoints, ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, format);
    }

    @Test
    public void writeCompressedOneOldFormatDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 1.1),
            point("2017-07-13T06:00:00Z", 2.1),
            point("2017-07-13T07:00:00Z", 3.3),
            point("2017-07-13T08:00:00Z", -3),
            point("2017-07-13T09:00:00Z", 8),
            point("2017-07-13T10:00:00Z", 12)
        );

        StockpileFormat format = StockpileFormat.byNumber(client.getCompatibleCompressFormat().lowerEndpoint());
        checkWriteOneCompressed(metricId, expectedPoints, ru.yandex.solomon.model.protobuf.MetricType.DGAUGE, format);
    }

    @Test
    public void writeCompressedOneLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED);
        List<TPoint> expectedPoints = Arrays.asList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(2d, 1d, 0d, 0d, 5d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build(),

            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:12:13Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(10)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(1d, 0d, 3d, 2d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build()
        );

        checkWriteOneCompressed(metricId, expectedPoints, ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM, StockpileFormat.CURRENT);
    }

    @Test
    public void writeCompressedOneWithUnsupportedFormat() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setBinaryVersion(100500)
            .addChunks(TimeSeries.Chunk.newBuilder()
                .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setToMillis(timeToMillis("2017-07-13T06:00:00Z"))
                .setPointCount(1000)
                .setContent(ByteString.copyFromUtf8("binary data with version that unsupported by stockpile"))
                .build()
            )
            .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.UNSUPPORTED_BINARY_FORMAT));
    }

    @Test
    public void writeCompressedOneCorrupted() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            // I not ensure that it's a good idea recheck binary data during writes
            // because in that case we lost all benefits from compressed data
            // and process cpu-bound uncompress and compress again
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .addChunks(TimeSeries.Chunk.newBuilder()
                .setFromMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setToMillis(timeToMillis("2017-07-13T06:00:00Z"))
                .setPointCount(1000)
                .setContent(ByteString.copyFromUtf8("Corrupted binary data"))
                .build()
            )
            .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatusMessage(), response.getStatus(), equalTo(EStockpileStatusCode.CORRUPTED_BINARY_DATA));
    }

    @Test
    public void readCompressedOneNewFormatDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 1.1),
            point("2017-07-13T06:00:00Z", 2.1),
            point("2017-07-13T07:00:00Z", 3.3),
            point("2017-07-13T08:00:00Z", -3),
            point("2017-07-13T09:00:00Z", 8),
            point("2017-07-13T10:00:00Z", 12)
        );

        StockpileFormat format = StockpileFormat.byNumber(client.getCompatibleCompressFormat().upperEndpoint());
        checkReadCompressedOne(metricId, expectedPoints, ColumnFlagMask.MINIMAL_DOUBLE_MASK, format);
    }

    @Test
    public void readCompressedOneNoData() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE)
                .setColumnMask(ColumnFlagMask.MINIMAL_LONG_MASK)
                .addPoints(TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setLongValue(42)
                        .build())
                .build());

        TCompressedReadResponse response = syncReadCompressedOne(TReadRequest.newBuilder()
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .setFromMillis(System.currentTimeMillis())
            .setToMillis(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(10))
            .setMetricId(metricId)
            .build());

        assertEquals(EStockpileStatusCode.OK, response.getStatus());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE, response.getType());
        assertThat(response.getBinaryVersion(), equalTo(StockpileFormat.CURRENT.getFormat()));

        assertEquals(response.getChunksCount(), 0);
    }

    @Test
    public void readCompressedManyNoData() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setType(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE)
                .setColumnMask(ColumnFlagMask.MINIMAL_LONG_MASK)
                .addPoints(TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setLongValue(42)
                        .build())
                .build());

        TCompressedReadManyResponse response = client.readCompressedMany(TReadManyRequest.newBuilder()
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .setFromMillis(System.currentTimeMillis())
            .setToMillis(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(10))
            .setShardId(metricId.getShardId())
            .addLocalIds(metricId.getLocalId())
            .build())
            .join();

        assertEquals(EStockpileStatusCode.OK, response.getStatus());

        var data = response.getMetrics(0);

        assertEquals(metricId.getLocalId(), data.getLocalId());
        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.IGAUGE, data.getType());
        assertTrue(data.hasCompressed());
    }

    @Test
    public void readCompressedOneOldFormatDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> expectedPoints = Arrays.asList(
            point("2017-07-13T05:00:00Z", 945),
            point("2017-07-13T06:00:00Z", 123),
            point("2017-07-13T07:00:00Z", 44),
            point("2017-07-13T08:00:00Z", 41),
            point("2017-07-13T09:00:00Z", 8.4),
            point("2017-07-13T10:00:00Z", 123)
        );

        StockpileFormat format = StockpileFormat.byNumber(client.getCompatibleCompressFormat().lowerEndpoint());
        checkReadCompressedOne(metricId, expectedPoints, ColumnFlagMask.MINIMAL_DOUBLE_MASK, format);
    }

    @Test
    public void readCompressedOneLogHistogram() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED);
        List<TPoint> expectedPoints = Arrays.asList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(5)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(2d, 1d, 0d, 0d, 5d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build(),

            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-13T05:10:00Z"))
                .setLogHistogram(LogHistogram.newBuilder()
                    .setZeroes(10)
                    .setStartPower(2)
                    .addAllBuckets(Arrays.asList(1d, 0d, 3d, 2d))
                    .setMaxBucketsSize(10)
                    .setBase(1)
                    .build())
                .build()
        );

        checkReadCompressedOne(metricId, expectedPoints, ColumnFlagMask.MINIMAL_LOG_HISTOGRAM_MASK, StockpileFormat.CURRENT);
    }

    @Test
    public void readCompressedOneTimeRangeAndDownSamplingDouble() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-13T05:00:00Z", 1))
            .addPoints(point("2017-07-13T06:00:00Z", 2))
            .addPoints(point("2017-07-13T07:00:00Z", 3))
            .addPoints(point("2017-07-13T07:45:12Z", 4))
            .addPoints(point("2017-07-13T07:58:41Z", 5))
            .addPoints(point("2017-07-13T08:00:00Z", 6))
            .addPoints(point("2017-07-13T09:00:00Z", 7))
            .addPoints(point("2017-07-13T10:00:00Z", 8))
            .build()
        );

        // previous format to guarantee that result encodes as we want
        StockpileFormat format = StockpileFormat.MIN;

        TCompressedReadResponse response = syncReadCompressedOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setBinaryVersion(format.getFormat())
            .setFromMillis(timeToMillis("2017-07-13T06:00:00Z")) // inclusive
            .setToMillis(timeToMillis("2017-07-13T09:00:00Z")) // exclusive
            .setGridMillis(TimeUnit.HOURS.toMillis(1L))
            .setAggregation(Aggregation.SUM)
            .build());

        List<TPoint> expectPoints = Arrays.asList(
            point("2017-07-13T06:00:00Z", 2, 1),
            point("2017-07-13T07:00:00Z", 3 + 4 + 5, 3),
            point("2017-07-13T08:00:00Z", 6, 1)
        );

        assertThat(response.getBinaryVersion(), equalTo(format.getFormat()));
        assertThat(uncompress(format, response.getChunksList()), equalTo(expectPoints));
    }

    @Test
    public void multiThreadCreateWriteOneReadOne() {
        List<MetricId> metrics = IntStream.range(0, 1000).parallel()
            .mapToObj(index -> syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE))
            .collect(Collectors.toList());

        metrics.parallelStream()
            .map(metricId -> client.writeOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                .addPoints(point("2017-07-13T06:00:00Z", 123))
                .addPoints(point("2017-07-13T06:15:00Z", metricId.getShardId()))
                .addPoints(point("2017-07-13T06:30:00Z", metricId.getLocalId()))
                .build())
            ).collect(Collectors.toList())
            .forEach(future -> ensureWriteSuccess(future.join()));

        List<TReadResponse> reads = metrics.parallelStream()
            .map(metricId -> client.readOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build()))
            .collect(Collectors.toList())
            .stream()
            .map(future -> {
                TReadResponse response = future.join();
                ensureReadSuccess(response);
                return response;
            })
            .collect(Collectors.toList());

        for (TReadResponse response : reads) {
            List<TPoint> expectedPoints = Arrays.asList(
                point("2017-07-13T06:00:00Z", 123),
                point("2017-07-13T06:15:00Z", response.getMetricId().getShardId()),
                point("2017-07-13T06:30:00Z", response.getMetricId().getLocalId())
            );

            assertThat(response.getPointsList(), equalTo(expectedPoints));
        }
    }

    @Test
    public void replacePartOfData() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        List<TPoint> sourcePoints = Arrays.asList(
            point("2017-07-03T06:01:21Z", 1),
            point("2017-07-03T06:02:30Z", 2),
            point("2017-07-03T06:03:51Z", 3),
            point("2017-07-03T06:05:00Z", 4),
            point("2017-07-13T06:12:00Z", 5),
            point("2017-07-13T06:23:00Z", 6),
            point("2017-07-13T06:34:00Z", 7)
        );

        List<TPoint> expect = Arrays.asList(
            point("2017-07-03T06:00:00Z", 1 + 2 + 3),
            point("2017-07-03T06:05:00Z", 4),
            point("2017-07-13T06:10:00Z", 5),
            point("2017-07-13T06:20:00Z", 6),
            point("2017-07-13T06:30:00Z", 7)
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addAllPoints(sourcePoints)
            .build());

        syncDeleteMetricData(DeleteMetricDataRequest.newBuilder()
            .setMetricId(metricId)
            .setToMillis(timeToMillis("2017-07-13T00:00:00Z")) // exclusive
            .build());

        // TODO: looks not useful
        // Without complete merge process we can't write data to delete space
        shards.forEach(StockpileShard::forceSnapshotSync);

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-03T06:01:00Z", 1 + 2 + 3))
            .addPoints(point("2017-07-03T06:05:00Z", 4))
            .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertEquals(expect, response.getPointsList());
    }

    @Test
    public void bulkShardCommandWrites() {
        final int shardId = 3;
        MetricId firstmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .build())
            .build());

        MetricId secondmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .build())
            .build());

        syncBulkShardCommand(TShardCommandRequest.newBuilder()
            .setShardId(shardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .setMetricId(firstmetricId)
                    .addPoints(point("2017-07-13T05:00:00Z", 1))
                    .addPoints(point("2017-07-13T06:00:00Z", 2))
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .setMetricId(secondmetricId)
                    .addPoints(point("2017-07-13T08:00:00Z", 6))
                    .addPoints(point("2017-07-13T09:00:00Z", 7))
                    .build())
                .build())
            .build());

        TReadResponse responseFirst = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(firstmetricId)
            .build());

        assertThat(responseFirst.getPointsList(),
            equalTo(Arrays.asList(
                point("2017-07-13T05:00:00Z", 1),
                point("2017-07-13T06:00:00Z", 2))
            )
        );

        TReadResponse readSecond = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(secondmetricId)
            .buildPartial());

        assertThat(readSecond.getPointsList(),
            equalTo(Arrays.asList(
                point("2017-07-13T08:00:00Z", 6),
                point("2017-07-13T09:00:00Z", 7))
            )
        );
    }

    @Test
    public void bulkShardCommandSummaryDouble() {
        final int shardId = 3;
        final long localId = Long.parseUnsignedLong("14856176262517368874");

        List<TPoint> expectedPoints = List.of(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2019-02-25T03:30:00Z"))
                .setSummaryDouble(SummaryDouble.newBuilder()
                    .setCount(1)
                    .setSum(1))
                .build(),

            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2019-02-25T03:35:00Z"))
                .setSummaryDouble(SummaryDouble.newBuilder()
                    .setCount(2)
                    .setSum(1))
                .build()
        );

        syncBulkShardCommand(TShardCommandRequest.newBuilder()
            .setShardId(shardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setMetricId(metricId(shardId, localId))
                    .setColumnMask(513)
                    .setType(ru.yandex.solomon.model.protobuf.MetricType.DSUMMARY)
                    .addAllPoints(expectedPoints)
                )
            )
            .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId(shardId, localId))
            .build());

        assertEquals(ru.yandex.solomon.model.protobuf.MetricType.DSUMMARY, response.getType());
        assertEquals(expectedPoints.size(), response.getPointsCount());
    }

    @Test
    public void bulkShardCommandsHistogram() {
        final int shardId = 3;
        MetricId firstmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
                .setType(ru.yandex.solomon.model.protobuf.MetricType.HIST)
                .setMetricId(MetricId.newBuilder()
                        .setShardId(shardId)
                        .build())
                .build());

        MetricId secondmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
                .setType(ru.yandex.solomon.model.protobuf.MetricType.HIST)
                .setMetricId(MetricId.newBuilder()
                        .setShardId(shardId)
                        .build())
                .build());

        List<TPoint> firstPoints = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:15:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:30:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 50d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 20L, 50L, 0L))
                                .build())
                        .build()
        );

        List<TPoint> secondPoints = Collections.singletonList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-13T05:00:00Z"))
                        .setHistogram(Histogram.newBuilder()
                                .addAllBounds(Arrays.asList(10d, 100d, 1000d))
                                .addAllBuckets(Arrays.asList(1L, 50L, 0L))
                                .build())
                        .build()
        );

        syncBulkShardCommand(TShardCommandRequest.newBuilder()
                .setShardId(3)
                .addCommands(TCommandRequest.newBuilder()
                        .setWrite(TWriteRequest.newBuilder()
                                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                                .setMetricId(firstmetricId)
                                .addAllPoints(firstPoints)
                                .build())
                        .build())
                .addCommands(TCommandRequest.newBuilder()
                        .setWrite(TWriteRequest.newBuilder()
                                .setColumnMask(ColumnFlagMask.MINIMAL_HISTOGRAM)
                                .setMetricId(secondmetricId)
                                .addAllPoints(secondPoints)
                                .build())
                        .build())
                .build());

        TReadResponse first = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(firstmetricId)
                .build());
        assertThat(first.getPointsList(), equalTo(firstPoints));

        TReadResponse second = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(secondmetricId)
                .build());
        assertThat(second.getPointsList(), equalTo(secondPoints));
    }

    @Test
    public void bulkShardCommandCompressedWrites() {
        MetricId originalmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setMetricId(MetricId.newBuilder()
                .setShardId(3)
                .build())
            .build());

        MetricId fiveMinmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setMetricId(MetricId.newBuilder()
                .setShardId(3)
                .build())
            .build());

        MetricId thirtyMinmetricId = syncCreateMetric(CreateMetricRequest.newBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
            .setMetricId(MetricId.newBuilder()
                .setShardId(3)
                .build())
            .build());

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(originalmetricId)
            .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
            .addPoints(point("2017-07-03T06:00:00Z", 1))
            .addPoints(point("2017-07-03T06:05:00Z", 2))
            .addPoints(point("2017-07-03T06:10:00Z", 3))
            .addPoints(point("2017-07-03T06:15:00Z", 4))
            .addPoints(point("2017-07-03T07:00:00Z", 5))
            .build());

        TCompressedReadResponse fiveMinReadCompres = syncReadCompressedOne(TReadRequest.newBuilder()
            .setMetricId(originalmetricId)
            .setAggregation(Aggregation.SUM)
            .setFromMillis(timeToMillis("2017-07-03T06:00:00Z"))
            .setToMillis(timeToMillis("2017-07-03T07:00:00Z"))
            .setGridMillis(TimeUnit.MINUTES.toMillis(5L))
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .build());

        TCompressedReadResponse thirtyMinReadCompres = syncReadCompressedOne(TReadRequest.newBuilder()
            .setMetricId(originalmetricId)
            .setAggregation(Aggregation.SUM)
            .setFromMillis(timeToMillis("2017-07-03T06:00:00Z"))
            .setGridMillis(TimeUnit.MINUTES.toMillis(30L))
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .build());

        syncBulkShardCommand(TShardCommandRequest.newBuilder()
            .setShardId(3)
            .addCommands(TCommandRequest.newBuilder()
                .setCompressedWrite(TCompressedWriteRequest.newBuilder()
                    .setMetricId(fiveMinmetricId)
                    .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                    .addAllChunks(fiveMinReadCompres.getChunksList())
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setCompressedWrite(TCompressedWriteRequest.newBuilder()
                    .setMetricId(thirtyMinmetricId)
                    .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                    .addAllChunks(thirtyMinReadCompres.getChunksList())
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setDeleteMetricData(DeleteMetricDataRequest.newBuilder()
                    .setMetricId(originalmetricId)
                    .setToMillis(timeToMillis("2017-07-03T07:00:00Z"))
                    .build())
                .buildPartial())
            .build());

        {
            TReadResponse orignRead = syncReadOne(TReadRequest.newBuilder()
                    .setMetricId(originalmetricId)
                    .setFromMillis(timeToMillis("2017-07-03T06:00:00Z"))
                    .setToMillis(timeToMillis("2017-07-03T07:00:00Z"))
                    .buildPartial());

            assertThat(orignRead.getPointsCount(), equalTo(0));
        }

        {
            TReadResponse fimeMinRead = syncReadOne(TReadRequest.newBuilder()
                    .setMetricId(fiveMinmetricId)
                    .setFromMillis(timeToMillis("2017-07-03T06:00:00Z"))
                    .setToMillis(timeToMillis("2017-07-03T07:00:00Z"))
                    .build());

            var expected = Arrays.asList(
                    point("2017-07-03T06:00:00Z", 1, 1),
                    point("2017-07-03T06:05:00Z", 2, 1),
                    point("2017-07-03T06:10:00Z", 3, 1),
                    point("2017-07-03T06:15:00Z", 4, 1));
            assertEquals(expected, dropNan(fimeMinRead.getPointsList()));
        }

        {
            TReadResponse thirtyMinRead = syncReadOne(TReadRequest.newBuilder()
                    .setMetricId(thirtyMinmetricId)
                    .setFromMillis(timeToMillis("2017-07-03T06:00:00Z"))
                    .setToMillis(timeToMillis("2017-07-03T07:00:00Z"))
                    .buildPartial());


            assertThat(dropNan(thirtyMinRead.getPointsList()),
                    equalTo(Collections.singletonList(
                            point("2017-07-03T06:00:00Z", 1 + 2 + 3 + 4, 4)
                            )
                    )
            );
        }
    }

    @Test
    public void writeOneDoubleWithAdditional() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);

        int columnMask = ColumnFlagMask.mask(
            EColumnFlag.COLUMN_TIMESTAMP,
            EColumnFlag.COLUMN_DOUBLE,
            EColumnFlag.COLUMN_MERGE,
            EColumnFlag.COLUMN_COUNT,
            EColumnFlag.COLUMN_STEP
        );

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(columnMask)
            .addPoints(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:30:00Z"))
                .setDoubleValue(1)
                .setMerge(true)
                .setCount(1)
                .setStepMillis(TimeUnit.MINUTES.toSeconds(1))
                .build())
            .build());

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(columnMask)
            .addPoints(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:30:00Z"))
                .setDoubleValue(2)
                .setMerge(true)
                .setCount(1)
                .setStepMillis(TimeUnit.MINUTES.toSeconds(1))
                .build())
            .build());

        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(columnMask)
            .addPoints(TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:30:00Z"))
                .setDoubleValue(3)
                .setMerge(true)
                .setCount(1)
                .setStepMillis(TimeUnit.MINUTES.toMillis(1))
                .build())
            .build());

        TReadResponse readResponse = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        List<TPoint> expectedPoints = Collections.singletonList(
            TPoint.newBuilder()
                .setTimestampsMillis(timeToMillis("2017-07-12T13:30:00Z"))
                .setDoubleValue(1 + 2 + 3)
                .setMerge(false)
                .setCount(3)
                .setStepMillis(TimeUnit.MINUTES.toMillis(5))
                .build()
        );

        assertEquals(expectedPoints, readResponse.getPointsList());
    }

    @Test
    public void writeEmptySummary() {
        MetricId metricId = syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType.DSUMMARY);

        List<TPoint> source = Arrays.asList(
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:40:00Z"))
                        .setSummaryDouble(SummaryDouble.getDefaultInstance())
                        .build(),
                TPoint.newBuilder()
                        .setTimestampsMillis(timeToMillis("2017-07-12T13:45:00Z"))
                        .setSummaryDouble(SummaryDouble.newBuilder()
                                .setCount(1)
                                .build())
                        .build());

        syncWriteOne(TWriteRequest.newBuilder()
                .setMetricId(metricId)
                .setColumnMask(ColumnFlagMask.MINIMAL_DSUMMARY)
                .addAllPoints(source)
                .build());

        TReadResponse response = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(metricId)
                .build());

        assertThat(response.getPointsList(), equalTo(source));
    }

    @Test
    public void allocateLocalIds() {
        var results = IntStream.range(0, 1000)
            .parallel()
            .mapToObj(ignore -> client.allocateLocalIds(TAllocateLocalIdsRequest.newBuilder()
                .setShardId(2)
                .setSize(100)
                .build()))
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .join();

        var unique = new LongOpenHashSet();
        for (var result : results) {
            assertEquals(EStockpileStatusCode.OK, result.getStatus());
            for (long localId : result.getList().getLocalIdsList()) {
                assertTrue(unique.add(localId));
            }
        }
    }

    @Test
    public void writeLog() {
        int shardId = 1;
        long localId = StockpileLocalId.random();
        var data = randomData();
        try (var log = log(localId, data)) {
            var writeResponse = client.writeLog(prepareWriteLogs(shardId, log)).join();
            assertEquals(writeResponse.getStatusMessage(), EStockpileStatusCode.OK, writeResponse.getStatus());
            var readResponse = syncReadOne(TReadRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(shardId)
                    .setLocalId(localId)
                    .build())
                .setToMillis(data.getTsMillis(data.length() - 1) + 1)
                .build());

            AggrGraphDataArrayList readResult = new AggrGraphDataArrayList();
            int mask = readResponse.getColumnMask();
            for (var proto : readResponse.getPointsList()) {
                readResult.addRecordData(mask, TPointConverters.fromProto(mask, proto));
            }

            assertEquals(data, readResult);
        }
    }

    private AggrGraphDataArrayList randomData() {
        int mask = StockpileColumns.minColumnSet(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE) | StepColumn.mask;
        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            var timeseries = new AggrGraphDataArrayList(mask, 10);
            var random = ThreadLocalRandom.current();
            for (int index = 0; index < 10; index++) {
                point.setTsMillis(TsRandomData.randomTs(random));
                point.setValue(ValueRandomData.randomNum(random));
                point.setStepMillis(10_000);
                timeseries.addRecord(point);
            }
            timeseries.sortAndMerge();
            return timeseries;
        } finally {
            point.recycle();
        }
    }

    private Log log(long localId, AggrGraphDataArrayList timeseries) {
        var random = ThreadLocalRandom.current();
        var alg = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        var buffer = ByteBufAllocator.DEFAULT;
        int numId = random.nextInt();
        try (var metaBuilder = new ResolvedLogMetaBuilderImpl(new ResolvedLogMetaHeader(numId, alg).setDecimPolicy(EDecimPolicy.POLICY_KEEP_FOREVER), buffer);
             var dataBuilder = new SnapshotLogDataBuilderImpl(alg, numId, buffer))
        {
            var archive = MetricArchiveImmutable.of(timeseries);
            int size = dataBuilder.onTimeSeries(archive);
            metaBuilder.onMetric(MetricType.DGAUGE, localId, archive.getRecordCount(), size);

            var meta = metaBuilder.build();
            var data = dataBuilder.build();
            return new Log(numId, meta, data);
        }
    }

    private TWriteLogRequest prepareWriteLogs(int shardId, Log... logs) {
        LogsIndex index = new LogsIndex(logs.length);
        ByteString content = ByteString.EMPTY;
        for (var log : logs) {
            index.add(log.numId, log.meta.readableBytes(), log.data.readableBytes());
            content = content.concat(ByteStrings.fromByteBuf(log.meta)).concat(ByteStrings.fromByteBuf(log.data));
        }

        ByteBuf serializedIndex = LogsIndexSerializer.serialize(UnpooledByteBufAllocator.DEFAULT, index);
        return TWriteLogRequest.newBuilder()
            .setShardId(shardId)
            .setIndex(ByteStrings.fromByteBuf(serializedIndex))
            .setContent(content)
            .build();
    }

    private void checkReadCompressedOne(MetricId metricId, List<TPoint> points, int columnSetMask, StockpileFormat format) {
        syncWriteOne(TWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setColumnMask(columnSetMask)
            .addAllPoints(points)
            .build());

        TCompressedReadResponse response = syncReadCompressedOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .setBinaryVersion(format.getFormat())
            .build());

        assertThat(response.getBinaryVersion(), equalTo(format.getFormat()));
        assertThat(uncompress(format, response.getChunksList()), equalTo(points));
    }

    private void checkWriteOneCompressed(MetricId metricId, List<TPoint> points, ru.yandex.solomon.model.protobuf.MetricType type, StockpileFormat format) {
        TCompressedWriteRequest request = TCompressedWriteRequest.newBuilder()
            .setMetricId(metricId)
            .setBinaryVersion(format.getFormat())
            .setType(type)
            .addAllChunks(compress(format, type, points))
            .build();

        TCompressedWriteResponse response = client.writeCompressedOne(request).join();
        assertThat(response.getStatus(), equalTo(EStockpileStatusCode.OK));

        TReadResponse readResponse = syncReadOne(TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build());

        assertThat(readResponse.getPointsList(), equalTo(points));
    }

    private List<TimeSeries.Chunk> compress(StockpileFormat format, ru.yandex.solomon.model.protobuf.MetricType type, List<TPoint> points) {
        final int mask = StockpileColumns.minColumnSet(type);

        AggrGraphDataArrayList source = new AggrGraphDataArrayList(mask, points.size());
        AggrPoint temp = new AggrPoint();
        for (TPoint point : points) {
            temp.setTsMillis(point.getTimestampsMillis());
            temp.setValue(point.getDoubleValue(), ValueColumn.DEFAULT_DENOM);
            temp.setLogHistogram(LogHistogramConverter.fromProto(point.getLogHistogram()));

            source.addRecordData(mask, temp);
        }

        var immutable = MetricArchiveImmutable.of(source);
        ByteString data = MetricArchiveNakedSerializer.serializerForFormatSealed(format)
                .serializeToByteString(immutable);

        LongSummaryStatistics stat = points.stream()
            .mapToLong(TPoint::getTimestampsMillis)
            .summaryStatistics();

        TimeSeries.Chunk chunk = TimeSeries.Chunk.newBuilder()
            .setContent(data)
            .setPointCount(immutable.getRecordCount())
            .setFromMillis(stat.getMin())
            .setToMillis(stat.getMax())
            .build();

        immutable.close();
        return Collections.singletonList(chunk);
    }

    private List<TPoint> uncompress(StockpileFormat format, List<TimeSeries.Chunk> chunks) {
        List<TPoint> result = new ArrayList<>(chunks.stream().mapToInt(TimeSeries.Chunk::getPointCount).sum());
        for (TimeSeries.Chunk chunk : chunks) {
            MetricArchiveImmutable archive = MetricArchiveNakedSerializer.serializerForFormatSealed(format)
                .deserializeToEof(new StockpileDeserializer(chunk.getContent()));

            result.addAll(Converters.toProto(archive.iterator()).getPointsList());
            archive.close();
        }

        return result;
    }

    private void deleteMetricAndCheck(MetricId metricId) {
        DeleteMetricRequest deleteRequest = DeleteMetricRequest.newBuilder()
            .setMetricId(metricId)
            .build();

        DeleteMetricResponse expectDeleteResponse = DeleteMetricResponse.newBuilder()
            .setStatus(EStockpileStatusCode.OK)
            .build();

        DeleteMetricResponse response = client.deleteMetric(deleteRequest).join();
        assertThat(response, equalTo(expectDeleteResponse));

        TReadRequest readRequest = TReadRequest.newBuilder()
            .setMetricId(metricId)
            .build();

        TReadResponse readResponse = client.readOne(readRequest).join();
        assertThat(readResponse.getPointsList(), equalTo(Collections.emptyList()));
    }

    private MetricId syncCreateMetric(ru.yandex.solomon.model.protobuf.MetricType type) {
        var list = shards.stream().collect(Collectors.toList());
        int idx = ThreadLocalRandom.current().nextInt(list.size());
        return MetricId.newBuilder()
                .setShardId(list.get(idx).shardId)
                .setLocalId(StockpileLocalId.random())
                .build();
    }

    private void syncWriteOne(TWriteRequest request) {
        TWriteResponse response = client.writeOne(request).join();
        ensureWriteSuccess(response);
    }

    private void ensureWriteSuccess(TWriteResponse response) {
        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }
    }

    private TReadResponse syncReadOne(TReadRequest request) {
        TReadResponse response = client.readOne(request).join();
        ensureReadSuccess(response);
        return response;
    }

    private void ensureReadSuccess(TReadResponse response) {
        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }
    }

    private TCompressedReadResponse syncReadCompressedOne(TReadRequest request) {
        TCompressedReadResponse response = client.readCompressedOne(request).join();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }

        return response;
    }

    private void syncDeleteMetric(DeleteMetricRequest request) {
        DeleteMetricResponse response = client.deleteMetric(request).join();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }
    }

    private MetricId syncCreateMetric(CreateMetricRequest request) {
        var id = request.getMetricId();
        MetricId generated = syncCreateMetric(request.getType());
        return MetricId.newBuilder()
                .setShardId(id.getShardId() != 0 ? id.getShardId() : generated.getShardId())
                .setLocalId(id.getLocalId() != 0 ? id.getLocalId() : generated.getLocalId())
                .build();
    }

    private void syncDeleteMetricData(DeleteMetricDataRequest request) {
        DeleteMetricDataResponse response = client.deleteMetricData(request).join();
        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }
    }

    private void syncBulkShardCommand(TShardCommandRequest request) {
        TShardCommandResponse response = client.bulkShardCommand(request).join();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ":" + response.getStatusMessage());
        }
    }

    private List<TPoint> dropNan(List<TPoint> points) {
        return points.stream()
            .filter(point -> !Double.isNaN(point.getDoubleValue()) && point.getDoubleValue() != 0)
            .collect(Collectors.toList());
    }
}
