package ru.yandex.stockpile.api.grpc;

import org.junit.Test;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.stockpile.api.TPoint;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.stockpile.api.grpc.TPointConverters.fromProto;
import static ru.yandex.stockpile.api.grpc.TPointConverters.toProto;

/**
 * @author Vladimir Gordiychuk
 */
public class TPointConvertersTest {
    @Test
    public void ts() {
        AggrPoint point = new AggrPoint();
        point.setTsMillis(System.currentTimeMillis());
        checkConvert(StockpileColumn.TS.mask(), point);
    }

    @Test
    public void doubleValue() {
        AggrPoint point = new AggrPoint();
        point.setValue(42, ValueColumn.DEFAULT_DENOM);
        checkConvert(StockpileColumn.VALUE.mask(), point);
    }

    @Test
    public void merge() {
        AggrPoint point = new AggrPoint();
        point.setMerge(true);
        checkConvert(StockpileColumn.MERGE.mask(), point);
        point.setMerge(false);
        checkConvert(StockpileColumn.MERGE.mask(), point);
    }

    @Test
    public void stepMillis() {
        AggrPoint point = new AggrPoint();
        point.setStepMillis(35_000);
        checkConvert(StockpileColumn.STEP.mask(), point);
    }

    @Test
    public void setHistogram() {
        AggrPoint point = new AggrPoint();
        point.setHistogram(AggrPoints.dhistogram(new double[]{10, 20, 30}, new long[]{2, 31, 0}));
        checkConvert(StockpileColumn.HISTOGRAM.mask(), point);
    }

    @Test
    public void setLogHistogram() {
        AggrPoint point = new AggrPoint();
        point.setLogHistogram(LogHistogram.ofBuckets(1, 2, 31, 4));
        checkConvert(StockpileColumn.LOG_HISTOGRAM.mask(), point);
    }

    @Test
    public void summaryInt64() {
        AggrPoint point = new AggrPoint();
        point.setSummaryInt64(AggrPoints.summaryInt64(12, 100, 2, 1));
        checkConvert(StockpileColumn.ISUMMARY.mask(), point);
    }

    @Test
    public void summaryDouble() {
        AggrPoint point = new AggrPoint();
        point.setSummaryDouble(AggrPoints.summaryDouble(1d, 43d, 21d));
        checkConvert(StockpileColumn.DSUMMARY.mask(), point);
    }

    @Test
    public void random() {
        for (int index = 0; index < 100; index++) {
            AggrPoint source = randomPoint();
            if (StockpileColumn.VALUE.isInSet(source.columnSet)) {
                source.valueDenom = ValueColumn.DEFAULT_DENOM;
            }

            checkConvert(source.columnSet, source);
        }
    }

    private void checkConvert(int mask, AggrPoint source) {
        TPoint proto = toProto(mask, source);
        AggrPoint converted = fromProto(mask, proto);
        assertEquals("Proto: " + proto.toString(), source, converted);
    }
}
