package ru.yandex.stockpile.api.grpc;

import com.google.protobuf.CodedInputStream;
import org.junit.Test;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.stockpile.api.TPoint;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.stockpile.api.grpc.TPointConverters.toProto;
import static ru.yandex.stockpile.api.grpc.TPointParser.parse;

/**
 * @author Vladimir Gordiychuk
 */
public class TPointParserTest {

    @Test
    public void randomPointParse() {
        for (int index = 0; index < 100; index++) {
            AggrPoint expected = randomPoint();
            // denom unsupported by proto
            expected.valueDenom = ValueColumn.DEFAULT_DENOM;
            TPoint proto = toProto(expected.columnSet, expected);

            AggrPoint parsed = serializeAndParse(proto);
            parsed.columnSet = expected.columnSet;
            assertEquals(expected, parsed);
        }
    }

    @Test
    public void tsParse() {
        AggrPoint expected = AggrPoint.builder()
            .time(System.currentTimeMillis())
            .build();

        TPoint proto = TPoint.newBuilder()
            .setTimestampsMillis(expected.tsMillis)
            .build();

        AggrPoint parsed = serializeAndParse(proto);
        assertEquals(expected, parsed);
    }

    @Test
    public void doubleValueParse() {
        AggrPoint expected = AggrPoint.builder()
            .doubleValue(42)
            .build();

        TPoint proto = TPoint.newBuilder()
            .setDoubleValue(42)
            .build();

        AggrPoint parsed = serializeAndParse(proto);
        assertEquals(expected, parsed);
    }

    @Test
    public void countParse() {
        AggrPoint expected = AggrPoint.builder()
            .count(123)
            .build();

        TPoint proto = TPoint.newBuilder()
            .setCount(123)
            .build();

        AggrPoint parsed = serializeAndParse(proto);
        assertEquals(expected, parsed);
    }

    @Test
    public void mergedParse() {
        AggrPoint expected = AggrPoint.builder()
            .merge(true)
            .build();

        TPoint proto = TPoint.newBuilder()
            .setMerge(true)
            .build();

        AggrPoint parsed = serializeAndParse(proto);
        assertEquals(expected, parsed);
    }

    private AggrPoint serializeAndParse(TPoint proto) {
        byte[] serialized = proto.toByteArray();
        return parse(CodedInputStream.newInstance(serialized));
    }
}
