package ru.yandex.stockpile.api.grpc.handler.parse;

import java.util.Random;

import com.google.protobuf.ByteString;
import com.google.protobuf.CodedInputStream;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.OwnerField;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequestBuilder;

import static junit.framework.TestCase.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class TWriteDataBinaryRequestParserTest {

    private final Random rnd = new Random();

    @Test
    public void parse() {
        EProjectId projectId = OwnerField.random(rnd);
        var builder = new StockpileShardWriteRequestBuilder(projectId, 42);

        long localId = StockpileLocalId.random();
        MetricType type = randomMetricType();
        AggrPoint point = randomPoint(type);
        builder.addRecord(localId, point, 1, type);
        byte[] bytes;
        try (var request = builder.build()) {
            bytes = request.serialize();
        }

        CodedInputStream input = TWriteDataBinaryRequest.newBuilder()
            .setShardId(44)
            .setContent(ByteString.copyFrom(bytes))
            .build()
            .toByteString()
            .newCodedInput();

        WriteDataBinaryRequest result = TWriteDataBinaryRequestParser.parse(input);
        assertEquals(44, result.shardId);

        MetricArchiveMutable archive = result.content.get(localId);
        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerProjectId(projectId.getNumber());
        expected.setOwnerShardId(42);
        expected.setDecimPolicyId((short) 1);
        expected.setType(type);
        expected.addRecord(point);

        assertEquals(expected, archive);
        close(archive, expected);
    }

    @Test
    public void negativeShardNumId() {
        EProjectId projectId = OwnerField.random(rnd);
        var builder = new StockpileShardWriteRequestBuilder(projectId, -1141592490);

        long localId = StockpileLocalId.random();
        MetricType type = randomMetricType();
        AggrPoint point = randomPoint(type);
        builder.addRecord(localId, point, (short) 1, type);
        byte[] bytes;
        try (var request = builder.build()) {
            bytes = request.serialize();
        }

        CodedInputStream input = TWriteDataBinaryRequest.newBuilder()
            .setShardId(44)
            .setContent(ByteString.copyFrom(bytes))
            .build()
            .toByteString()
            .newCodedInput();

        WriteDataBinaryRequest result = TWriteDataBinaryRequestParser.parse(input);
        assertEquals(44, result.shardId);

        MetricArchiveMutable archive = result.content.get(localId);
        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerProjectId(projectId.getNumber());
        expected.setOwnerShardId(-1141592490);
        expected.setDecimPolicyId((short) 1);
        expected.setType(type);
        expected.addRecord(point);

        assertEquals(expected, archive);
        close(expected, archive);
    }
}
