package ru.yandex.stockpile.api.grpc.mem;

import java.util.Objects;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.SummaryDouble;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.DeleteMetricDataRequest;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.client.ColumnFlagMask;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static ru.yandex.stockpile.client.TestUtil.metricId;
import static ru.yandex.stockpile.client.TestUtil.point;
import static ru.yandex.stockpile.client.TestUtil.timeToMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardBatchRequestMarshallerTest {
    private ByteBuf buffer;
    private ShardBatchRequestMarshaller marshaller;

    @Before
    public void setUp() throws Exception {
        marshaller = new ShardBatchRequestMarshaller();
    }

    @After
    public void tearDown() throws Exception {
        if (buffer != null) {
            buffer.release();
            buffer = null;
        }
    }

    @Test
    public void shardIdDeserialize() throws Exception {
        final int expectShardId = 112;
        buffer = toByteBuf(TShardCommandRequest.newBuilder()
            .setShardId(expectShardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setMetricId(MetricId.newBuilder()
                        .setShardId(expectShardId)
                        .setLocalId(123)
                        .build())
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addPoints(point("2017-07-12T13:30:00Z", 3))
                    .addPoints(point("2017-07-12T13:20:00Z", 2))
                    .addPoints(point("2017-07-12T13:10:00Z", 1))
                    .build())
                .build())
            .build());

        try (ShardBatchRequest request = marshaller.parse(new ByteBufInputStream(buffer))) {
            assertThat(request.getShardId(), equalTo(expectShardId));
        }
    }

    @Test
    public void emptyShardCommandRequest() throws Exception {
        final int expectShardId = 333;
        buffer = toByteBuf(TShardCommandRequest.newBuilder()
            .setShardId(expectShardId)
            .build());

        try (ShardBatchRequest request = marshaller.parse(new ByteBufInputStream(buffer))) {
            assertThat(request.getShardId(), equalTo(expectShardId));
            assertThat(request.countRecords(), equalTo(0L));
        }
    }

    @Test
    public void countPointsInBatch() throws Exception {
        final int expectShardId = 444;
        buffer = toByteBuf(TShardCommandRequest.newBuilder()
            .setShardId(expectShardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 111))
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addPoints(point("2017-07-12T13:30:00Z", 3))
                    .addPoints(point("2017-07-12T13:20:00Z", 2))
                    .addPoints(point("2017-07-12T13:10:00Z", 1))
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setDeleteMetric(DeleteMetricRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 222))
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setDeleteMetricData(DeleteMetricDataRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 333))
                    .setToMillis(System.currentTimeMillis())
                    .build())
                .build())
            .build());

        long expectedRecordCount = 3 + 2; // delete commands should be also included
        try (ShardBatchRequest request = marshaller.parse(new ByteBufInputStream(buffer))) {
            assertThat(request.countRecords(), equalTo(expectedRecordCount));
        }
    }

    @Test
    public void summaryDouble() {
        final int shardId = 2234;
        final long localId = Long.parseUnsignedLong("14856176262517368874");
        buffer = toByteBuf(TShardCommandRequest.newBuilder()
            .setShardId(shardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setMetricId(metricId(shardId, localId))
                    .setColumnMask(513)
                    .setType(MetricType.DSUMMARY)
                    .addPoints(TPoint.newBuilder()
                        .setTimestampsMillis(1551065505000L)
                        .setSummaryDouble(SummaryDouble.newBuilder()
                            .setCount(1)
                            .setSum(1)
                            .build())
                        .build())
                    .addPoints(TPoint.newBuilder()
                        .setTimestampsMillis(1551065510000L)
                        .setSummaryDouble(SummaryDouble.newBuilder()
                            .setCount(2)
                            .setSum(1)
                            .build())
                        .build())
                    .build())
                .build())
            .setDeadline(1551069174783L)
            .build());

        try (ShardBatchRequest request = marshaller.parse(new ByteBufInputStream(buffer))) {
            assertFalse(request.isEmpty());
            assertEquals(2, request.countRecords());
            assertEquals(shardId, request.getShardId());

            var content = request.getContent();
            assertNotNull(content);
            assertFalse(content.isEmpty());
            assertEquals(3, content.getRecordsInArchives());

            var logEntryTwo = request.getContent();
            assertFalse(logEntryTwo.isEmpty());
            assertEquals(3, logEntryTwo.getRecordsInArchives());
        }
    }

    @Test
    public void appendToLog() throws Exception {
        final int expectShardId = 444;
        buffer = toByteBuf(TShardCommandRequest.newBuilder()
            .setShardId(expectShardId)
            .addCommands(TCommandRequest.newBuilder()
                .setWrite(TWriteRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 111))
                    .setColumnMask(ColumnFlagMask.MINIMAL_DOUBLE_MASK)
                    .addPoints(point("2017-07-12T13:30:00Z", 3))
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setDeleteMetric(DeleteMetricRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 222))
                    .build())
                .build())
            .addCommands(TCommandRequest.newBuilder()
                .setDeleteMetricData(DeleteMetricDataRequest.newBuilder()
                    .setMetricId(metricId(expectShardId, 333))
                    .setToMillis(timeToMillis("2017-07-12T13:20:00Z"))
                    .build())
                .build())
            .build());

        try (ShardBatchRequest request = marshaller.parse(new ByteBufInputStream(buffer))) {
            var content = Objects.requireNonNull(request.getContent());
            Long2ObjectOpenHashMap<MetricArchiveMutable> data = content.getArchiveByLocalId();

            assertThat(data.get(333).getDeleteBefore(), equalTo(timeToMillis("2017-07-12T13:20:00Z")));
            assertThat(data.get(222).getDeleteBefore(), equalTo(DeleteBeforeField.DELETE_ALL));
            assertThat(data.get(111).toAggrGraphDataArrayList(), equalTo(AggrGraphDataArrayList.listShort(timeToMillis("2017-07-12T13:30:00Z"), 3d)));
        }
    }

    private ByteBuf toByteBuf(TShardCommandRequest request) {
        return Unpooled.wrappedBuffer(request.toByteArray()).asReadOnly();
    }
}
