package ru.yandex.stockpile.cluster.balancer.dao;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.CommonResource;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;
import ru.yandex.solomon.balancer.snapshot.SnapshotNode;
import ru.yandex.solomon.balancer.snapshot.SnapshotShard;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class StockpileBalancerDaoTest {

    @Before
    public void setUp() {
        getDao().createSchema().join();
    }

    @Test
    public void defaultAssignmentState() {
        var state = getDao().getAssignments().join();
        assertNotNull(state);
    }

    @Test
    public void defaultOpts() {
        var opts = getDao().getOptions().join();
        assertNotNull(opts);
    }

    @Test
    public void saveReadOpts() {
        Resources limits = new Resources();
        limits.set(CommonResource.CPU, 50);
        limits.set(CommonResource.MEMORY, 10 << 10);
        limits.set(CommonResource.RECORDS_WRITE_RATE, 0);
        limits.set(CommonResource.METRICS_READ_RATE, 0);
        limits.set(CommonResource.SHARDS_COUNT, 100);
        var expected = BalancerOptions.newBuilder()
                .setVersion(100500)
                .setGracefulUnassignExpiration(5, TimeUnit.SECONDS)
                .setForceUnassignExpiration(1, TimeUnit.SECONDS)
                .setLimits(limits)
                .build();

        getDao().saveOptions(expected).join();
        var result = getDao().getOptions().join();
        assertEquals(expected, result);
    }

    @Test
    public void saveReadAssignmentState() {
        var node = new SnapshotNode();
        node.address = "solomon-test";
        node.freeze = true;
        node.active = true;

        var shard = new SnapshotShard();
        shard.shardId = "42";
        shard.node = "solomon-test";
        shard.resources = new Resources();
        shard.resources.set(CommonResource.CPU, 0.2);
        shard.resources.set(CommonResource.MEMORY, 1 << 10);
        shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 5.1);
        shard.resources.set(CommonResource.METRICS_READ_RATE, 10.5);

        var snapshot = new SnapshotAssignments(List.of(node), List.of(shard));

        getDao().saveAssignments(snapshot).join();
        var result = getDao().getAssignments().join();
        assertEquals(snapshot, result);
    }

    @Test
    public void replaceOpts() {
        Resources limits = new Resources();
        limits.set(CommonResource.CPU, 50);
        limits.set(CommonResource.MEMORY, 10 << 10);
        limits.set(CommonResource.METRICS_READ_RATE, 0);
        limits.set(CommonResource.RECORDS_WRITE_RATE, 0);
        limits.set(CommonResource.SHARDS_COUNT, 100);
        var v1 = BalancerOptions.newBuilder()
                .setVersion(100500)
                .setGracefulUnassignExpiration(5, TimeUnit.SECONDS)
                .setForceUnassignExpiration(1, TimeUnit.SECONDS)
                .setLimits(limits)
                .build();

        getDao().saveOptions(v1).join();

        var v2 = v1.toBuilder()
            .setVersion(100501)
            .build();

        getDao().saveOptions(v2).join();
        var result = getDao().getOptions().join();
        assertEquals(v2, result);
    }

    @Test
    public void replaceAssignments() {
        SnapshotAssignments v1;
        {
            var node = new SnapshotNode();
            node.address = "bob";
            node.freeze = false;
            node.active = true;

            var shard = new SnapshotShard();
            shard.shardId = "42";
            shard.node = "bob";
            shard.resources = new Resources();
            shard.resources.set(CommonResource.CPU, 0.5);
            shard.resources.set(CommonResource.MEMORY, 1 << 10);
            shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 5.1);
            shard.resources.set(CommonResource.METRICS_READ_RATE, 10.5);

            v1 = new SnapshotAssignments(List.of(node), List.of(shard));
        }

        getDao().saveAssignments(v1).join();

        SnapshotAssignments v2;
        {
            var shard = new SnapshotShard();
            shard.shardId = "43";
            shard.node = "bob";
            shard.resources = new Resources();
            shard.resources.set(CommonResource.CPU, 0.6);
            shard.resources.set(CommonResource.MEMORY, 2 << 10);
            shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 10.1);
            shard.resources.set(CommonResource.METRICS_READ_RATE, 20.2);

            v2 = new SnapshotAssignments(v1.nodes, List.of(v1.shards.get(0), shard));
        }

        getDao().saveAssignments(v2).join();
        var result = getDao().getAssignments().join();
        assertEquals(v2, result);
    }

    protected abstract BalancerDao getDao();
}
