package ru.yandex.stockpile.kikimrKv;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.kikimr.client.KikimrGrpcTransport;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientImpl;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
@YaIgnore
@Ignore
public class KvAlterTableTest {

    private KikimrKvClient kvClient;

    @Before
    public void setUp() {
        kvClient = new KikimrKvClientImpl(
                new KikimrGrpcTransport(List.of(HostAndPort.fromString("localhost:25791")), 10 << 20));
    }

    @After
    public void tearDown() throws Exception {
        kvClient.close();
    }

    @Test(expected = CompletionException.class)
    public void alterNotExists() {
        kvClient.alterKvTablets(nextPath(), 10).join();
    }

    @Test
    public void increaseAmountSame() {
        var path = nextPath();
        kvClient.createKvTablets(path, 10).join();
        var created = kvClient.resolveKvTablets(path).join();

        kvClient.alterKvTablets(path, 10).join();

        var altered = kvClient.resolveKvTablets(path).join();
        assertArrayEquals(created, altered);
    }

    @Test(expected = CompletionException.class)
    public void decreaseAmount() {
        var path = nextPath();
        kvClient.createKvTablets(path, 10).join();
        kvClient.alterKvTablets(path, 5).join();
    }

    @Test
    public void increaseSaveOrder() {
        var path = nextPath();
        kvClient.createKvTablets(path, 5).join();
        var prevLocalIds = kvClient.resolveKvTablets(path).join();

        for (int index = 0; index < 20; index++) {
            var resolvedOne = kvClient.resolveKvTablets(path).join();
            System.out.println(Arrays.toString(resolvedOne));
            assertArrayEquals(prevLocalIds, resolvedOne);

            var expectedSize = prevLocalIds.length + (index * 5);
            kvClient.alterKvTablets(path, expectedSize).join();
            var resolvedTwo = kvClient.resolveKvTablets(path).join();
            assertEquals(expectedSize, resolvedTwo.length);
            assertArrayEquals(prevLocalIds, Arrays.copyOf(resolvedTwo, prevLocalIds.length));
            prevLocalIds = resolvedTwo;
        }
    }

    @Test
    public void increasePlusNoise() {
        var path = nextPath();
        kvClient.createKvTablets(path, 5).join();
        var prevLocalIds = kvClient.resolveKvTablets(path).join();

        for (int index = 0; index < 20; index++) {
            var resolvedOne = kvClient.resolveKvTablets(path).join();
            System.out.println(Arrays.toString(resolvedOne));
            assertArrayEquals(prevLocalIds, resolvedOne);

            kvClient.createKvTablets(nextPath(), index + 1).join();

            var expectedSize = prevLocalIds.length + (index * 5);
            kvClient.alterKvTablets(path, expectedSize).join();
            var resolvedTwo = kvClient.resolveKvTablets(path).join();
            assertEquals(expectedSize, resolvedTwo.length);
            assertArrayEquals(prevLocalIds, Arrays.copyOf(resolvedTwo, prevLocalIds.length));
            prevLocalIds = resolvedTwo;
        }
    }


    @Test
    public void tryToWriteToAlteredTablets() {
        var path = nextPath();
        kvClient.createKvTablets(path, 5).join();
        var localIdsV1 = kvClient.resolveKvTablets(path).join();

        for (var localId : localIdsV1) {
            var file = "c." + localId;
            var content = "Hi " + localId;
            write(localId, file, content);
        }

        kvClient.alterKvTablets(path, 10).join();
        var localIdsV2 = kvClient.resolveKvTablets(path).join();

        for (var localId : localIdsV1) {
            var file = "c." + localId;
            var content = "Hi " + localId;
            var files = listFiles(localId);
            assertEquals(List.of(file), files);
            assertEquals(content, readFile(localId, file));
            write(localId, file + 2, file);
        }

        for (int index = 5; index < 10; index++) {
            var localId = localIdsV2[index];
            var file = "c." + localId;
            var content = "Hi " + localId;
            var files = listFiles(localId);
            assertEquals(List.of(), files);
            write(localId, file, content);
            assertEquals(content, readFile(localId, file));
        }
    }

    private void write(long localId, String file, String text) {
        kvClient.write(
                localId,
                0,
                file,
                text.getBytes(StandardCharsets.UTF_8),
                MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
                MsgbusKv.TKeyValueRequest.EPriority.REALTIME,
                0)
                .join();
    }

    private List<String> listFiles(long localId) {
        return kvClient.readRangeNames(localId, 0, NameRange.all(), 0).join()
                .stream()
                .map(KikimrKvClient.KvEntryStats::getName)
                .collect(Collectors.toList());
    }

    @Nullable
    private String readFile(long localId, String file) {
        return kvClient.readData(localId, 0, file, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME)
                .join()
                .map(bytes -> {
                    try {
                        return new String(bytes, StandardCharsets.UTF_8.name());
                    } catch (UnsupportedEncodingException e) {
                        return null;
                    }
                })
                .orElse(null);
    }

    private String nextPath() {
        return "/local/" + UUID.randomUUID().toString();
    }
}
