package ru.yandex.stockpile.kikimrKv;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryStats;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EPriority;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.stockpile.server.SnapshotLevel;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class KvFilesTest {

    private KikimrKvClient kvClient;
    private ShardIdMapToLong tabletMapping;
    private KvFiles kvFiles;

    @Before
    public void before() {
        kvClient = new KikimrKvClientInMem();
        kvClient.createKvTablets("/test", 10).join();
        tabletMapping = new ShardIdMapToLong(kvClient.resolveKvTablets("/test").join());
        kvFiles = new KvFiles(kvClient, "/test");
    }

    @Test
    public void deleteSnapshotIndexChunkCommand() {
        int shardId = 1;
        writeFile(shardId, "c.i2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00001");
        writeFile(shardId, "c.c2.00000000046960054.00001");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.ETERNITY, 46960054).join();

        var expected = List.of(
                "d.c.c2.00000000046960054.00001",
                "d.c.i2.00000000046960054.00001",
                "d.c.s2.00000000046960054.00001");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void deleteSnapshotAllChunks() {
        int shardId = 2;
        writeFile(shardId, "c.s1.00000000046960055.00001");
        writeFile(shardId, "c.s1.00000000046960055.00002");
        writeFile(shardId, "c.s1.00000000046960055.00003");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();
        var expected = List.of(
                "d.c.s1.00000000046960055.00001",
                "d.c.s1.00000000046960055.00002",
                "d.c.s1.00000000046960055.00003");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void deleteSnapshotOnlyTarget() {
        int shardId = 2;
        writeFile(shardId, "c.s1.00000000046960054.00001");
        writeFile(shardId, "c.s1.00000000046960055.00001");
        writeFile(shardId, "c.s1.00000000046960056.00001");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();
        System.out.println(listFiles(shardId));
        var expected = List.of(
                "c.s1.00000000046960054.00001",
                "c.s1.00000000046960056.00001",
                "d.c.s1.00000000046960055.00001");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreSnapshotIndexChunkCommand() {
        int shardId = 1;
        writeFile(shardId, "c.i2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00001");
        writeFile(shardId, "c.c2.00000000046960054.00001");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.ETERNITY, 46960054).join();
        kvFiles.restoreSnapshot(shardId, SnapshotLevel.ETERNITY, 46960054).join();

        var expected = List.of(
                "c.c2.00000000046960054.00001",
                "c.i2.00000000046960054.00001",
                "c.s2.00000000046960054.00001");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreSnapshotAllChunks() {
        int shardId = 2;
        writeFile(shardId, "c.s1.00000000046960055.00001");
        writeFile(shardId, "c.s1.00000000046960055.00002");
        writeFile(shardId, "c.s1.00000000046960055.00003");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();
        kvFiles.restoreSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();

        var expected = List.of(
                "c.s1.00000000046960055.00001",
                "c.s1.00000000046960055.00002",
                "c.s1.00000000046960055.00003");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreSnapshotOnlyTarget() {
        int shardId = 2;
        writeFile(shardId, "c.s1.00000000046960054.00001");
        writeFile(shardId, "c.s1.00000000046960055.00001");
        writeFile(shardId, "c.s1.00000000046960056.00001");

        kvFiles.deleteSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();
        kvFiles.restoreSnapshot(shardId, SnapshotLevel.DAILY, 46960055).join();
        System.out.println(listFiles(shardId));
        var expected = List.of(
                "c.s1.00000000046960054.00001",
                "c.s1.00000000046960055.00001",
                "c.s1.00000000046960056.00001");
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void deleteSnapshotByFile() {
        int shardId = 1;
        writeFile(shardId, "c.i2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00002");
        writeFile(shardId, "c.c2.00000000046960054.00001");
        writeFile(shardId, "c.i2.00000000046960055.00001");

        kvFiles.deleteFile(shardId, "c.s2.00000000046960054.00002").join();

        var expected = List.of(
                "c.i2.00000000046960055.00001",
                "d.c.c2.00000000046960054.00001",
                "d.c.i2.00000000046960054.00001",
                "d.c.s2.00000000046960054.00001",
                "d.c.s2.00000000046960054.00002"
        );
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreSnapshotByFile() {
        int shardId = 1;
        writeFile(shardId, "c.i2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00001");
        writeFile(shardId, "c.s2.00000000046960054.00002");
        writeFile(shardId, "c.c2.00000000046960054.00001");
        writeFile(shardId, "c.i2.00000000046960055.00001");

        kvFiles.deleteFile(shardId, "c.s2.00000000046960054.00002").join();
        kvFiles.restoreFile(shardId, "c.s2.00000000046960054.00002").join();

        var expected = List.of(
                "c.c2.00000000046960054.00001",
                "c.i2.00000000046960054.00001",
                "c.i2.00000000046960055.00001",
                "c.s2.00000000046960054.00001",
                "c.s2.00000000046960054.00002"
                );
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void deleteLog() {
        int shardId = 1;
        writeFile(shardId, "c.l.00000000024585603.00001y");
        writeFile(shardId, "c.l.00000000024585603.00002z");

        kvFiles.deleteLog(shardId, 24585603).join();

        var expected = List.of(
                "d.c.l.00000000024585603.00001y",
                "d.c.l.00000000024585603.00002z"
        );
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreLog() {
        int shardId = 1;
        writeFile(shardId, "c.l.00000000024585603.00001y");
        writeFile(shardId, "c.l.00000000024585603.00002z");

        kvFiles.deleteLog(shardId, 24585603).join();
        kvFiles.restoreLog(shardId, 24585603).join();

        var expected = List.of(
                "c.l.00000000024585603.00001y",
                "c.l.00000000024585603.00002z"
        );
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void deleteLogByFile() {
        int shardId = 1;
        writeFile(shardId, "c.l.00000000024585603.00001y");
        writeFile(shardId, "c.l.00000000024585603.00002z");

        kvFiles.deleteFile(shardId, "c.l.00000000024585603.00002z").join();

        var expected = List.of(
                "d.c.l.00000000024585603.00001y",
                "d.c.l.00000000024585603.00002z"
        );
        assertEquals(expected, listFiles(shardId));
    }

    @Test
    public void restoreLogByFile() {
        int shardId = 1;
        writeFile(shardId, "c.l.00000000024585603.00001y");
        writeFile(shardId, "c.l.00000000024585603.00002z");

        kvFiles.deleteFile(shardId, "c.l.00000000024585603.00002z").join();
        kvFiles.restoreFile(shardId, "c.l.00000000024585603.00002z").join();

        var expected = List.of(
                "c.l.00000000024585603.00001y",
                "c.l.00000000024585603.00002z"
        );
        assertEquals(expected, listFiles(shardId));
    }

    private List<String> listFiles(int shardId) {
        var tabletId = tabletMapping.get(shardId);
        var files = kvClient.readRangeNames(tabletId, 0, 0).join();
        return files.stream().map(KvEntryStats::getName).collect(Collectors.toList());
    }

    private void writeFile(int shardId, String file) {
        var tabletId = tabletMapping.get(shardId);
        byte[] bytes = new byte[100];
        ThreadLocalRandom.current().nextBytes(bytes);
        kvClient.write(tabletId, 0, file, bytes, EStorageChannel.MAIN, EPriority.REALTIME, 0).join();
    }
}
