package ru.yandex.stockpile.kikimrKv;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.protobuf.ByteString;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.SnapshotCommandPartsSerialized;
import ru.yandex.stockpile.server.data.index.SnapshotIndexPartsSerialized;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentSerializer;
import ru.yandex.stockpile.server.data.log.StockpileLogEntrySerialized;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.file.LogFile;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;


/**
 * @author Sergey Polovko
 */
public class KvStockpileShardStorageTest {
    private final Random random = new Random();
    private KikimrKvClientInMem kvClient;
    private KikimrKvClientCounting kvClientCounting;
    private KvStockpileShardStorage storage;

    @Before
    public void before() {
        kvClient = new KikimrKvClientInMem();
        long tabletId = kvClient.createKvTablet();

        kvClientCounting = new KikimrKvClientCounting(kvClient, new KikimrKvClientMetrics(new MetricRegistry()));
        storage = new KvStockpileShardStorage(null, kvClientCounting, tabletId);
        storage.lock().join();
    }

    @Test
    public void smallLog() {
        long txn = 1;
        StockpileLogEntrySerialized logEntry = new StockpileLogEntrySerialized(
            txn,
            generateContent(100));

        syncWriteLog(logEntry);
        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals(1, files.length);

        Assert.assertEquals("c.l.00000000000000001.00000z", files[0].getName());
        Assert.assertEquals(logEntry.size(), files[0].getSize());
    }

    @Test
    public void hugeLog() {
        long txn = 2;
        StockpileLogEntrySerialized logEntry = new StockpileLogEntrySerialized(
            txn,
            generateContent(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE + 100));

        syncWriteLog(logEntry);
        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals(2, files.length);

        Assert.assertEquals("c.l.00000000000000002.00000y", files[0].getName());
        Assert.assertEquals("c.l.00000000000000002.00001z", files[1].getName());
        Assert.assertEquals(logEntry.size(), files[0].getSize() + files[1].getSize());
    }

    @Test
    public void smallSnapshot() {
        long txn = 3;
        StockpileLogEntrySerialized logEntry = new StockpileLogEntrySerialized(
            txn,
            generateContent(100));

        syncWriteSnapshot(txn, logEntry);
        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals(1, files.length);

        Assert.assertEquals("c.l.00000000000000003.00000z", files[0].getName());
        Assert.assertEquals(logEntry.size(), files[0].getSize());
    }

    @Test
    public void hugeSnapshot() {
        long txn = 4;
        StockpileLogEntrySerialized logEntry = new StockpileLogEntrySerialized(
            txn,
            generateContent(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE + 100));

        syncWriteSnapshot(txn, logEntry);
        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals(2, files.length);

        Assert.assertEquals("c.l.00000000000000004.00000y", files[0].getName());
        Assert.assertEquals("c.l.00000000000000004.00001z", files[1].getName());
        Assert.assertEquals(logEntry.size(), files[0].getSize() + files[1].getSize());
    }

    @Test
    public void logSnapshotWithDelete() {
        StockpileLogEntryContent snapshot = new StockpileLogEntryContent();

        long txn = 5;
        final long firstTxn = txn;
        final int logsCount = 100;

        // (1) write logs and collect snapshot
        for (int i = 0; i < logsCount; i++) {
            StockpileLogEntryContent logEntry = generateLogEntry(1000 + random.nextInt(10_000));
            try {
                snapshot.overrideWith(logEntry);

                ByteString bytes = StockpileLogEntryContentSerializer.S.serializeToByteString(logEntry);
                Assert.assertTrue(bytes.size() < KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE);
                syncWriteLog(new StockpileLogEntrySerialized(txn++, bytes));
            } finally {
                logEntry.release();
            }
        }

        Assert.assertEquals(logsCount, syncListFiles().length);

        // (2) write snapshot and delete logs
        StockpileLogEntrySerialized snapshotSerialized = new StockpileLogEntrySerialized(
            txn,
            StockpileLogEntryContentSerializer.S.serializeToByteString(snapshot));
        syncWriteSnapshot(firstTxn, snapshotSerialized);
        snapshot.release();

        // (3) check written files
        int snapshotSize = snapshotSerialized.size();
        int chunksCount = snapshotSize / KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE + 1;

        KikimrKvClient.KvEntryStats[] files = syncListFiles();
        Assert.assertEquals(chunksCount, files.length);

        int filesSize = 0;
        for (int i = 0; i < files.length; i++) {
            LogFile file = (LogFile) FileNameParsed.parseCurrent(files[i].getName());
            Assert.assertEquals(txn, file.txn());
            Assert.assertEquals(i, file.getPartNo());

            if (i == files.length - 1) {
                Assert.assertTrue(file.isLast());
            } else {
                Assert.assertFalse(file.isLast());
            }

            filesSize += files[i].getSize();
        }

        Assert.assertEquals(snapshotSize, filesSize);
    }

    @Test
    public void smallSnapshotChunkToTemp() {
        ChunkWithNo[] chunks = {
            new ChunkWithNo(0, "chunk0".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(1, "chunk1".getBytes(StandardCharsets.UTF_8)),
        };

        for (ChunkWithNo chunk : chunks) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.TWO_HOURS, 127, chunk).join();
        }

        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals("t.c.s0.00000000000000127.00000", files[0].getName());
        Assert.assertEquals(6, files[0].getSize());

        Assert.assertEquals("t.c.s0.00000000000000127.00001", files[1].getName());
        Assert.assertEquals(6, files[1].getSize());
    }

    @Test
    public void hugeSnapshotChunkToTemp() {
        ChunkWithNo[] chunks = {
            new ChunkWithNo(0, new byte[KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE - 1]),
            new ChunkWithNo(1, new byte[KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE]),
            new ChunkWithNo(2, new byte[KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE + 1]),
        };

        for (ChunkWithNo chunk : chunks) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.TWO_HOURS, 128, chunk).join();
        }

        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals("t.c.s0.00000000000000128.00000", files[0].getName());
        Assert.assertEquals(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE - 1, files[0].getSize());

        Assert.assertEquals("t.c.s0.00000000000000128.00001", files[1].getName());
        Assert.assertEquals(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE, files[1].getSize());

        Assert.assertEquals("t.c.s0.00000000000000128.00002", files[2].getName());
        Assert.assertEquals(KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE + 1, files[2].getSize());
    }

    @Test
    public void writeSnapshotIndexToTemp() {
        SnapshotIndexPartsSerialized parts = new SnapshotIndexPartsSerialized(
            SnapshotLevel.TWO_HOURS,
            129,
            new ByteString[] {
                ByteString.copyFrom("index0", StandardCharsets.UTF_8),
                ByteString.copyFrom("index1", StandardCharsets.UTF_8),
            });

        storage.writeSnapshotIndexToTemp(parts).join();

        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals("t.c.i0.00000000000000129.00000y", files[0].getName());
        Assert.assertEquals(6, files[0].getSize());

        Assert.assertEquals("t.c.i0.00000000000000129.00001z", files[1].getName());
        Assert.assertEquals(6, files[1].getSize());
    }

    @Test
    public void writeSnapshotCommandToTemp() {
        var parts = new SnapshotCommandPartsSerialized(
                SnapshotLevel.TWO_HOURS,
                129,
                new ByteString[] {
                        ByteString.copyFrom("command0", StandardCharsets.UTF_8),
                        ByteString.copyFrom("command1", StandardCharsets.UTF_8),
                });

        storage.writeSnapshotCommandToTemp(parts).join();

        KikimrKvClient.KvEntryStats[] files = syncListFiles();

        Assert.assertEquals("t.c.c0.00000000000000129.00000y", files[0].getName());
        Assert.assertEquals(parts.content()[0].size(), files[0].getSize());

        Assert.assertEquals("t.c.c0.00000000000000129.00001z", files[1].getName());
        Assert.assertEquals(parts.content()[1].size(), files[1].getSize());
    }

    @Test
    public void repeatedRename() {
        long txn = 42L;
        ChunkWithNo[] chunks = {
            new ChunkWithNo(0, "chunk0".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(1, "chunk1".getBytes(StandardCharsets.UTF_8)),
        };

        for (ChunkWithNo chunk : chunks) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.TWO_HOURS, txn, chunk).join();
        }

        // ensure temp files exists after complete write chunks
        {

            KikimrKvClient.KvEntryStats[] files = syncListFiles();
            Assert.assertEquals(2, files.length);

            Assert.assertEquals("t.c.s0.00000000000000042.00000", files[0].getName());
            Assert.assertEquals(6, files[0].getSize());

            Assert.assertEquals("t.c.s0.00000000000000042.00001", files[1].getName());
            Assert.assertEquals(6, files[1].getSize());
        }

        var address = new SnapshotAddress(SnapshotLevel.TWO_HOURS, txn);
        storage.renameSnapshotDeleteLogs(address).join();

        // ensure that temp file renamed to current
        {
            KikimrKvClient.KvEntryStats[] files = syncListFiles();
            Assert.assertEquals(2, files.length);

            Assert.assertEquals("c.s0.00000000000000042.00000", files[0].getName());
            Assert.assertEquals(6, files[0].getSize());

            Assert.assertEquals("c.s0.00000000000000042.00001", files[1].getName());
            Assert.assertEquals(6, files[1].getSize());
        }

        // same rename not affect anything
        storage.renameSnapshotDeleteLogs(address).join();
        KikimrKvClient.KvEntryStats[] files = syncListFiles();
        Assert.assertEquals(2, files.length);

        Assert.assertEquals("c.s0.00000000000000042.00000", files[0].getName());
        Assert.assertEquals(6, files[0].getSize());

        Assert.assertEquals("c.s0.00000000000000042.00001", files[1].getName());
        Assert.assertEquals(6, files[1].getSize());
    }

    @Test
    public void repeatedRename2() {
        long txn = 42L;
        ChunkWithNo[] chunks = {
            new ChunkWithNo(0, "chunk0".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(1, "chunk1".getBytes(StandardCharsets.UTF_8)),
        };

        for (ChunkWithNo chunk : chunks) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.DAILY, txn, chunk).join();
        }

        // ensure temp files exists after complete write chunks
        {

            KikimrKvClient.KvEntryStats[] files = syncListFiles();
            Assert.assertEquals(2, files.length);

            Assert.assertEquals("t.c.s1.00000000000000042.00000", files[0].getName());
            Assert.assertEquals(6, files[0].getSize());

            Assert.assertEquals("t.c.s1.00000000000000042.00001", files[1].getName());
            Assert.assertEquals(6, files[1].getSize());
        }

        var address = new SnapshotAddress(SnapshotLevel.DAILY, txn);
        storage.renameSnapshotDeleteOld(new SnapshotAddress[]{address}, new SnapshotAddress[0]).join();

        // ensure that temp file renamed to current
        {
            KikimrKvClient.KvEntryStats[] files = syncListFiles();
            Assert.assertEquals(2, files.length);

            Assert.assertEquals("c.s1.00000000000000042.00000", files[0].getName());
            Assert.assertEquals(6, files[0].getSize());

            Assert.assertEquals("c.s1.00000000000000042.00001", files[1].getName());
            Assert.assertEquals(6, files[1].getSize());
        }

        // same rename not affect anything
        storage.renameSnapshotDeleteLogs(address).join();
        KikimrKvClient.KvEntryStats[] files = syncListFiles();
        Assert.assertEquals(2, files.length);

        Assert.assertEquals("c.s1.00000000000000042.00000", files[0].getName());
        Assert.assertEquals(6, files[0].getSize());

        Assert.assertEquals("c.s1.00000000000000042.00001", files[1].getName());
        Assert.assertEquals(6, files[1].getSize());
    }

    @Test
    public void renameMultipleTmpIndex() {
        long txn = 42L;
        ChunkWithNo[] daily = {
            new ChunkWithNo(0, "dailyChunk0".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(1, "dailyChunk1".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(2, "dailyChunk2".getBytes(StandardCharsets.UTF_8)),
        };

        ChunkWithNo[] eternity = {
            new ChunkWithNo(0, "eternityChunk0".getBytes(StandardCharsets.UTF_8)),
            new ChunkWithNo(1, "eternityChunk1".getBytes(StandardCharsets.UTF_8)),
        };

        for (ChunkWithNo chunk : daily) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.DAILY, txn, chunk).join();
        }

        for (ChunkWithNo chunk : eternity) {
            storage.writeSnapshotChunkToTemp(SnapshotLevel.ETERNITY, txn, chunk).join();
        }

        // ensure temp files exists after complete write chunks
        {
            List<String> expected = Arrays.asList(
                // daily
                "t.c.s1.00000000000000042.00000",
                "t.c.s1.00000000000000042.00001",
                "t.c.s1.00000000000000042.00002",
                // eternity
                "t.c.s2.00000000000000042.00000",
                "t.c.s2.00000000000000042.00001"
            );


            var files = Stream.of(syncListFiles())
                .map(KikimrKvClient.KvEntryStats::getName)
                .collect(Collectors.toList());

            assertEquals(expected, files);
        }

        var snapshots = new SnapshotAddress[]{
            new SnapshotAddress(SnapshotLevel.DAILY, txn),
            new SnapshotAddress(SnapshotLevel.ETERNITY, txn),
        };
        storage.renameSnapshotDeleteOld(snapshots, new SnapshotAddress[0]).join();

        List<String> expected = Arrays.asList(
            // daily
            "c.s1.00000000000000042.00000",
            "c.s1.00000000000000042.00001",
            "c.s1.00000000000000042.00002",
            // eternity
            "c.s2.00000000000000042.00000",
            "c.s2.00000000000000042.00001"
        );

        // ensure that temp file renamed to current
        {
            var files = Stream.of(syncListFiles())
                .map(KikimrKvClient.KvEntryStats::getName)
                .collect(Collectors.toList());

            assertEquals(expected, files);
        }

        // same rename not affect anything
        storage.renameSnapshotDeleteOld(snapshots, new SnapshotAddress[0]).join();
        var files = Stream.of(syncListFiles())
            .map(KikimrKvClient.KvEntryStats::getName)
            .collect(Collectors.toList());

        assertEquals(expected, files);
    }

    @Test
    public void lockIncrementGeneration() {
        long tabletId = kvClient.createKvTablet();
        long tableGen = kvClient.getGeneration(tabletId);

        storage = new KvStockpileShardStorage(null, kvClientCounting, tabletId);
        storage.lock().join();
        assertNotEquals(tableGen, kvClient.getGeneration(tabletId));
    }

    @Test
    public void skipLockWhenGenerationAlreadyDefined() {
        long tabletId = kvClient.createKvTablet();
        long tableGen = kvClient.incrementGeneration(tabletId, 0).join();

        storage = new KvStockpileShardStorage(null, kvClientCounting, tabletId, tableGen);
        storage.lock().join();
        assertEquals(tableGen, kvClient.getGeneration(tabletId));
    }

    private void syncWriteSnapshot(long firstSnapshotTxn, StockpileLogEntrySerialized serialized) {
        storage.writeLogSnapshot(firstSnapshotTxn, serialized).join();
    }

    private void syncWriteLog(StockpileLogEntrySerialized serialized) {
        storage.writeLogEntry(serialized).join();
    }

    private KikimrKvClient.KvEntryStats[] syncListFiles() {
        return storage.readRangeNames(ReadClass.OTHER, NameRange.all())
            .join()
            .stream()
            .sorted(Comparator.comparing(KikimrKvClient.KvEntryStats::getName))
            .toArray(KikimrKvClient.KvEntryStats[]::new);
    }

    private ByteString generateContent(int maxSize) {
        var logEntry = new StockpileLogEntryContent();
        var point = RecyclableAggrPoint.newInstance();
        try {
            point.columnSet = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
            point.tsMillis = System.currentTimeMillis();

            long localId = 1;
            while (true) {
                for (int i = 0; i < 100_000; i++) {
                    fillNextRandomValue(point);
                    logEntry.addAggrPoint(localId++ & 0xffff, point);
                }

                var bytes = StockpileLogEntryContentSerializer.S.serializeToByteString(logEntry);
                if (bytes.size() > maxSize) {
                    return bytes;
                }
            }
        } finally {
            point.recycle();
            logEntry.release();
        }
    }

    private StockpileLogEntryContent generateLogEntry(int points) {
        StockpileLogEntryContent logEntry = new StockpileLogEntryContent();

        var point = RecyclableAggrPoint.newInstance();
        point.columnSet = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        point.tsMillis = System.currentTimeMillis();
        long localId = 1;
        for (int i = 0; i < points; i++) {
            fillNextRandomValue(point);
            logEntry.addAggrPoint(localId++ & 0xffff, point);
        }
        point.recycle();

        return logEntry;
    }

    private void fillNextRandomValue(AggrPoint point) {
        point.tsMillis += 15_000 + random.nextInt(180_000);
        point.valueNum = ValueRandomData.randomNum(random);
        point.valueDenom = ValueRandomData.randomDenom(random);
    }
}
