package ru.yandex.stockpile.memState;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.LongStream;

import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.tool.BitBufAssume;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class LogEntriesContentTest {

    @Test
    public void objectSizeEmpty() {
        LogEntriesContent content = new LogEntriesContent();

        GraphLayout gl = GraphLayout.parseInstance(content);
        assertEquals(gl.totalSize(), content.memorySizeIncludingSelf());
        content.release();
    }

    @Test
    public void objectSize() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        LogEntriesContent content = new LogEntriesContent();
        for (int index = 1; index < 100; index++) {
            StockpileLogEntryContent s = new StockpileLogEntryContent();
            s.addAggrPoint(index % 10, randomPoint(MetricType.DGAUGE));
            content.updateWithLogEntry(42, s);
        }

        // Exclude enums because exists into single instance into app
        long enumSize = GraphLayout.parseInstance(MetricType.DGAUGE, StockpileFormat.CURRENT).totalSize();

        GraphLayout gl = GraphLayout.parseInstance(content);
        assertEquals(gl.totalSize() - enumSize, content.memorySizeIncludingSelf());
        content.release();
    }

    @Test
    public void updateWithEmpty() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        LogEntriesContent target = new LogEntriesContent();
        target.updateWithLogEntriesContent(42, new LogEntriesContent());
        assertTrue(target.isEmpty());
        target.release();
    }

    @Test
    public void addNew() {
        long localId = StockpileLocalId.random();
        MetricToArchiveMap update = new MetricToArchiveMap();
        update.getArchiveRef(localId).addRecord(randomPoint());

        LogEntriesContent target = new LogEntriesContent();
        target.updateWithLogEntriesContent(41, new LogEntriesContent(update, new DeletedShardSet(), new Int2LongOpenHashMap(), 1, 128));

        assertEquals(1, target.getFileCount());
        assertEquals(128, target.getFileSize());
        assertNotNull(target.getMetricToArchiveMap().getById(localId));
        target.release();
        update.release();
    }

    @Test
    public void addManyNew() {
        MetricToArchiveMap update = new MetricToArchiveMap();
        List<Long> localIds = new ArrayList<>();
        for (int index = 0; index < 30_000; index++) {
            long localId = StockpileLocalId.random();
            update.getArchiveRef(localId).addRecord(randomPoint());
            localIds.add(localId);
        }

        LogEntriesContent target = new LogEntriesContent();
        target.updateWithLogEntriesContent(41, new LogEntriesContent(update,  new DeletedShardSet(), new Int2LongOpenHashMap(), 2, 256));

        assertEquals(2, target.getFileCount());
        assertEquals(256, target.getFileSize());

        for (long localId : localIds) {
            assertNotNull(target.getMetricToArchiveMap().getById(localId));
        }
        update.release();
        target.release();
    }

    @Test
    public void updateMany() {
        long[] localIds = LongStream.generate(StockpileLocalId::random)
            .limit(1_000)
            .toArray();

        LogEntriesContent target = new LogEntriesContent();
        long now = System.currentTimeMillis();
        AggrPoint point = new AggrPoint();
        point.setTsMillis(now);
        for (int index = 0; index < 10; index++) {
            MetricToArchiveMap update = new MetricToArchiveMap();
            try {
                now += 10_000;
                point.setTsMillis(now);
                point.setValue(index);
                for (long localId : localIds) {
                    update.getArchiveRef(localId).addRecord(point);
                }

                update.getArchiveRef(StockpileLocalId.random()).addRecord(randomPoint());
                target.updateWithLogEntriesContent(42, new LogEntriesContent(update, new DeletedShardSet(), new Int2LongOpenHashMap(), 1, 10));
            } finally {
                update.release();
            }
        }


        MetricArchiveMutable archive = target.getMetricToArchiveMap().getById(localIds[0]);
        assertNotNull(archive);
        assertEquals(10, archive.getRecordCount());
        assertEquals(10, target.getFileCount());
        assertEquals(100, target.getFileSize());
        target.release();
    }

    @Test
    public void updateDeletedShards() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        LogEntriesContent target = new LogEntriesContent();
        DeletedShardSet deletedShards = new DeletedShardSet();
        deletedShards.add(1, 42);

        target.updateWithLogEntriesContent(42, new LogEntriesContent(new MetricToArchiveMap(), deletedShards, new Int2LongOpenHashMap(), 1, 10));
        assertFalse(target.isEmpty());
        assertTrue(target.getDeletedShards().contains(1, 42));
        target.release();
    }
}
