package ru.yandex.stockpile.server.cache;

import com.google.common.base.Preconditions;
import org.junit.Assert;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;


/**
 * @author Stepan Koltsov
 */
public class Long2ObjectLruCacheTest {

    static final class MeasurableInteger extends DefaultObject implements MemMeasurable, AutoCloseable {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MeasurableInteger.class);
        private final int value;
        private boolean closed;

        MeasurableInteger(int value) {
            this.value = value;
        }

        int getInt() {
            Preconditions.checkArgument(!closed, "closed");
            return value;
        }

        @Override
        public long memorySizeIncludingSelf() {
            Preconditions.checkArgument(!closed, "closed");
            return SELF_SIZE;
        }

        @Override
        public void close() {
            Preconditions.checkArgument(!closed, "closed");
            closed = true;
        }
    }

    static final class MeasurableString extends DefaultObject implements MemMeasurable, AutoCloseable {
        private final String value;
        private boolean closed;

        MeasurableString(String value) {
            this.value = value;
        }

        String getStr() {
            Preconditions.checkArgument(!closed, "closed");
            return value;
        }

        @Override
        public long memorySizeIncludingSelf() {
            Preconditions.checkArgument(!closed, "closed");
            return MemoryCounter.objectSelfSizeLayout(MeasurableString.class) +
                MemoryCounter.objectSelfSizeLayout(String.class) +
                value.length() * 2;
        }

        @Override
        public void close() {
            Preconditions.checkArgument(!closed, "closed");
            closed = true;
        }
    }

    @Test
    public void replace() {
        Long2ObjectLruCache<MeasurableString> cache = new Long2ObjectLruCache<>();
        long initSize = cache.getBytesUsed();

        cache.replace(10, new MeasurableString("aa"));
        long sizeOfOneItem = cache.getBytesUsed();
        Assert.assertTrue(sizeOfOneItem > initSize);

        cache.replace(20, new MeasurableString("xxx"));
        long sizeOfTwoItems = cache.getBytesUsed();
        Assert.assertTrue(sizeOfTwoItems > sizeOfOneItem);

        cache.replace(10, new MeasurableString("bbbb"));
        long sizeWithBiggerFirstItem = cache.getBytesUsed();
        Assert.assertTrue(sizeWithBiggerFirstItem > sizeOfTwoItems);
    }

    @Test
    public void remove() {
        Long2ObjectLruCache<MeasurableInteger> cache = new Long2ObjectLruCache<>();
        long initSize = cache.getBytesUsed();

        cache.replace(1, new MeasurableInteger(10));
        long sizeOfOneItem = cache.getBytesUsed();
        Assert.assertTrue(sizeOfOneItem > initSize);

        cache.remove(2);
        assertEquals(sizeOfOneItem, cache.getBytesUsed());

        cache.replace(2, new MeasurableInteger(20));
        long sizeOfTwoItems = cache.getBytesUsed();
        Assert.assertTrue(sizeOfTwoItems > sizeOfOneItem);

        cache.remove(1);
        assertEquals(sizeOfOneItem, cache.getBytesUsed());

        cache.remove(2);
        assertEquals(initSize, cache.getBytesUsed());
    }

    @Test
    public void objectSizeEmpty() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
    }

    @Test
    public void objectSize() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        for (int i = 0; i < 100; i++) {
            source.replace(i, new MeasurableInteger(i));
        }

        assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
    }

    @Test
    public void objectSizeAfterRemove() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        for (int i = 0; i < 25; i++) {
            source.replace(i, new MeasurableInteger(i));
            assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
        }

        for (int i = 0; i < 25; i++) {
            source.remove(i);
            assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
        }
    }

    @Test
    public void closeOnReplace() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        var one = new MeasurableInteger(1);
        var two = new MeasurableInteger(2);
        source.replace(42, one);
        assertFalse(one.closed);
        source.replace(42, two);
        assertTrue(one.closed);
        assertFalse(two.closed);
    }

    @Test
    public void closeOnClear() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        var one = new MeasurableInteger(1);
        source.replace(42, one);
        assertFalse(one.closed);
        source.clear();
        assertTrue(one.closed);
    }

    @Test
    public void closeOnClose() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        var one = new MeasurableInteger(1);
        source.replace(42, one);
        assertFalse(one.closed);
        source.close();
        assertTrue(one.closed);
    }

    @Test
    public void closeOnRemove() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        var one = new MeasurableInteger(1);
        source.replace(42, one);
        assertFalse(one.closed);
        source.remove(42);
        assertTrue(one.closed);
    }

    @Test
    public void closeOnRemoveOlder() {
        Long2ObjectLruCache<MeasurableInteger> source = new Long2ObjectLruCache<>();
        var one = new MeasurableInteger(1);
        source.replace(42, one);
        assertFalse(one.closed);
        source.setMaxSizeBytes(1);
        source.removeEntriesAboveSizeLimit();
        assertTrue(one.closed);
    }

    private long expectedLayout(MemMeasurable source) {
        GraphLayout gl = GraphLayout.parseInstance(source);
        return gl.totalSize();
    }
}
