package ru.yandex.stockpile.server.data.chunk;


import org.junit.Assert;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;

import static org.junit.Assert.assertEquals;

/**
 * @author Sergey Polovko
 */
public class ChunkIndexTest {

    @Test
    public void entries() {
        // empty
        {
            ChunkIndex index = new ChunkIndex();
            Assert.assertEquals(0, index.metricCount());
        }

        // one
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);

            Assert.assertEquals(1, index.metricCount());

            Assert.assertEquals(1, index.getLocalId(0));
            Assert.assertEquals(1000, index.getLastTssMillis(0));
            Assert.assertEquals(10, index.getSize(0));
            Assert.assertEquals(0, index.getOffset(0));
        }

        // many
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            index.addMetric(2, 2000, 20);
            index.addMetric(3, 3000, 30);

            Assert.assertEquals(3, index.metricCount());

            Assert.assertEquals(1, index.getLocalId(0));
            Assert.assertEquals(1000, index.getLastTssMillis(0));
            Assert.assertEquals(10, index.getSize(0));
            Assert.assertEquals(0, index.getOffset(0));

            Assert.assertEquals(2, index.getLocalId(1));
            Assert.assertEquals(2000, index.getLastTssMillis(1));
            Assert.assertEquals(20, index.getSize(1));
            Assert.assertEquals(10, index.getOffset(1));

            Assert.assertEquals(3, index.getLocalId(2));
            Assert.assertEquals(3000, index.getLastTssMillis(2));
            Assert.assertEquals(30, index.getSize(2));
            Assert.assertEquals(30, index.getOffset(2));
        }
    }

    @Test
    public void findMetricData() {
        // empty
        {
            ChunkIndex index = new ChunkIndex();
            Assert.assertNull(index.findMetricData(0));
            Assert.assertNull(index.findMetricData(1));
            Assert.assertNull(index.findMetricData(2));
        }

        // one
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            Assert.assertNull(index.findMetricData(0));

            DataRange range = index.findMetricData(1);
            Assert.assertNotNull(range);
            Assert.assertEquals(0, range.getOffset());
            Assert.assertEquals(10, range.getLength());

            Assert.assertNull(index.findMetricData(2));
        }

        // many
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            index.addMetric(2, 2000, 20);
            index.addMetric(3, 3000, 30);
            index.addMetric(-1, 4000, 40);
            Assert.assertNull(index.findMetricData(0));

            DataRange r1 = index.findMetricData(1);
            Assert.assertNotNull(r1);
            Assert.assertEquals(0, r1.getOffset());
            Assert.assertEquals(10, r1.getLength());

            DataRange r2 = index.findMetricData(2);
            Assert.assertNotNull(r2);
            Assert.assertEquals(10, r2.getOffset());
            Assert.assertEquals(20, r2.getLength());

            DataRange r3 = index.findMetricData(3);
            Assert.assertNotNull(r3);
            Assert.assertEquals(30, r3.getOffset());
            Assert.assertEquals(30, r3.getLength());

            DataRange r4 = index.findMetricData(-1);
            Assert.assertNotNull(r4);
            Assert.assertEquals(60, r4.getOffset());
            Assert.assertEquals(40, r4.getLength());

            Assert.assertNull(index.findMetricData(4));
        }
    }

    @Test
    public void findMetricLastTsMillis() {
        // empty
        {
            ChunkIndex index = new ChunkIndex();
            Assert.assertEquals(-1, index.findMetricLastTsMillis(0));
            Assert.assertEquals(-1, index.findMetricLastTsMillis(1));
            Assert.assertEquals(-1, index.findMetricLastTsMillis(2));
        }

        // one
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);

            Assert.assertEquals(-1, index.findMetricLastTsMillis(0));
            Assert.assertEquals(1000, index.findMetricLastTsMillis(1));
            Assert.assertEquals(-1, index.findMetricLastTsMillis(2));
        }

        // many
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            index.addMetric(2, 2000, 20);
            index.addMetric(3, 3000, 30);
            index.addMetric(-1, 4000, 40);

            Assert.assertEquals(-1, index.findMetricLastTsMillis(0));
            Assert.assertEquals(1000, index.findMetricLastTsMillis(1));
            Assert.assertEquals(2000, index.findMetricLastTsMillis(2));
            Assert.assertEquals(3000, index.findMetricLastTsMillis(3));
            Assert.assertEquals(4000, index.findMetricLastTsMillis(-1));
            Assert.assertEquals(-1, index.findMetricLastTsMillis(4));
        }
    }

    @Test
    public void getSizesArray() {
        // empty
        {
            ChunkIndex index = new ChunkIndex();
            Assert.assertArrayEquals(new int[] {}, index.getSizesArray());
        }

        // one
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            Assert.assertArrayEquals(new int[] { 10 }, index.getSizesArray());
            Assert.assertEquals(10, index.getSize(0));
        }

        // many
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            index.addMetric(2, 2000, 20);
            index.addMetric(3, 3000, 30);
            Assert.assertArrayEquals(new int[] { 10, 20, 30 }, index.getSizesArray());
            Assert.assertEquals(10, index.getSize(0));
            Assert.assertEquals(20, index.getSize(1));
            Assert.assertEquals(30, index.getSize(2));
        }
    }

    @Test
    public void countChunkDiskSize() {
        // empty
        {
            ChunkIndex index = new ChunkIndex();
            Assert.assertEquals(0, index.countChunkDiskSize());
        }

        // one
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            Assert.assertEquals(10, index.countChunkDiskSize());
        }

        // many
        {
            ChunkIndex index = new ChunkIndex();
            index.addMetric(1, 1000, 10);
            index.addMetric(2, 2000, 20);
            index.addMetric(3, 3000, 30);
            Assert.assertEquals(10 + 20 + 30, index.countChunkDiskSize());
        }
    }

    @Test
    public void objectSizeEmpty() {
        ChunkIndex source = new ChunkIndex(0);
        assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
    }

    @Test
    public void objectSize() {
        ChunkIndex source = new ChunkIndex();
        for (int index = 1; index < 95; index++) {
            source.addMetric(index, index * 1000, index * 10);
        }
        assertEquals(expectedLayout(source), source.memorySizeIncludingSelf());
    }

    private long expectedLayout(ChunkIndex source) {
        GraphLayout gl = GraphLayout.parseInstance(source);
        return gl.totalSize();
    }
}
