package ru.yandex.stockpile.server.data.chunk;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.shard.SnapshotReason;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class ChunkWriterTest {

    private SnapshotIndexContent index;
    private ChunkWriter writer;
    private List<ChunkWithNo> chunks;

    @Before
    public void setUp() {
        writer = new ChunkWriter(SnapshotReason.UNKNOWN, System.currentTimeMillis(), 0);
        chunks = new ArrayList<>();
    }

    @Test
    public void writeEmptyMutable() {
        MetricArchiveMutable empty = new MetricArchiveMutable();
        empty.setOwnerShardId(42);

        write(1L, empty);
        finish();

        MetricArchiveImmutable result = read(1L);
        var immutable = empty.toImmutable();
        assertThat(result, equalTo(immutable));
        close(empty, result, immutable);
    }

    @Test
    public void writeWithPointsMutable() {
        MetricArchiveMutable source = MetricArchiveMutable.of(
                AggrGraphDataArrayList.of(
                        point("2018-07-23T08:01:00Z", 1d),
                        point("2018-07-23T08:02:00Z", 2d),
                        point("2018-07-23T08:04:00Z", 4d),
                        point("2018-07-23T08:03:00Z", 3d),
                        point("2018-07-23T08:05:00Z", 5d)
                ));

        write(2L, source);
        finish();

        MetricArchiveImmutable result = read(2L);
        var immutable = source.toImmutable();
        assertThat(result, equalTo(immutable));
        assertThat(index.getRecordCount(), equalTo(5L));
        close(source, immutable);
    }

    @Test
    public void writeEmptyImmutable() {
        MetricArchiveMutable mutable = new MetricArchiveMutable();
        mutable.setOwnerShardId(42);
        MetricArchiveImmutable empty = mutable.toImmutable();

        write(4L, empty);
        finish();

        MetricArchiveImmutable result = read(4L);
        assertThat(result, equalTo(empty));
        close(mutable, empty, result);
    }

    @Test
    public void writeWithPointsImmutable() {
        MetricArchiveImmutable source = MetricArchiveMutable.of(
                AggrGraphDataArrayList.of(
                        point("2018-07-23T08:01:00Z", 1d),
                        point("2018-07-23T08:02:00Z", 2d),
                        point("2018-07-23T08:04:00Z", 4d),
                        point("2018-07-23T08:03:00Z", 3d),
                        point("2018-07-23T08:05:00Z", 5d)
                ))
                .toImmutable();

        write(3L, source);
        finish();

        MetricArchiveImmutable result = read(3L);
        assertThat(result, equalTo(source));
        assertThat(index.getRecordCount(), equalTo(5L));
        close(source, result);
    }

    @Test
    public void writeStatsByMetric() {
        int ownerId = -1662362008;
        MetricArchiveMutable source = MetricArchiveMutable.of(
            AggrGraphDataArrayList.of(
                point("2018-07-23T08:01:00Z", 1d),
                point("2018-07-23T08:02:00Z", 2d),
                point("2018-07-23T08:04:00Z", 4d),
                point("2018-07-23T08:03:00Z", 3d),
                point("2018-07-23T08:05:00Z", 5d)
            ));

        source.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        source.setOwnerShardId(ownerId);

        write(3L, source.toImmutable());
        finish();

        var stats = index.getStats()
            .getStatsByProject(EProjectId.SOLOMON)
            .getByOwner()
            .get(ownerId)
            .getStatsByKind(source.getType());
        assertEquals(5, stats.records);
        close(source);
    }

    @Test
    public void memoryUsage() {
        var initMemory = writer.memorySizeIncludingSelf();
        assertThat(initMemory, greaterThan(0L));

        MetricArchiveMutable source = MetricArchiveMutable.of(
            AggrGraphDataArrayList.of(
                point("2018-07-23T08:01:00Z", 1d),
                point("2018-07-23T08:02:00Z", 2d),
                point("2018-07-23T08:04:00Z", 4d),
                point("2018-07-23T08:03:00Z", 3d),
                point("2018-07-23T08:05:00Z", 5d)
            ));

        write(2L, source);

        assertThat(writer.memorySizeIncludingSelf(), greaterThan(initMemory));
        assertEquals(initMemory + source.bytesCount(), writer.memorySizeIncludingSelf(), 200);

        write(3L, source);
        write(4L, source);
        write(5L, source);

        assertEquals(initMemory + (source.bytesCount() * 4), writer.memorySizeIncludingSelf(), 200);

        for (int index = 6; index < 10000; index++) {
            write(index, source);
        }

        finish();
        close(source);
    }

    private MetricArchiveImmutable read(long metricId) {
        DataRangeInSnapshot range = index.findMetricData(metricId);
        assertThat("Not found metric with id: " + metricId, range, notNullValue());

        ChunkWithNo chunk = chunks.get(range.getChunkNo());
        DataRange dataRange = range.getDataRange();
        return MetricArchiveNakedSerializer.serializerForFormatSealed(index.getFormat())
                .deserializeRange(chunk.getContent(), dataRange.getOffset(), dataRange.getLength());
    }

    private void write(long metricId, MetricArchiveMutable archive) {
        ChunkWithNo chunk = writer.writeMetric(metricId, archive.getLastTsMillis(), archive.toImmutableNoCopy());
        if (chunk != null) {
            chunks.add(chunk);
        }
    }

    private void write(long metricId, MetricArchiveImmutable archive) {
        long latestTsMillis = archive.toMutable().getLastTsMillis();

        ChunkWithNo chunk = writer.writeMetric(metricId, latestTsMillis, archive);
        if (chunk != null) {
            chunks.add(chunk);
        }
    }

    private void finish() {
        ChunkWriter.Finish finish = writer.finish();
        chunks.add(finish.chunkWithNo);
        index = finish.index;
    }
}
