package ru.yandex.stockpile.server.data.dao;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.Try;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientSync;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestContext;

import static org.junit.Assert.assertNotNull;

/**
 * @author Stepan Koltsov
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
    StockpileShardTestContext.class,
    KikimrKvClientSync.class
})
public class ReadBatcherTest {

    @Autowired
    private ReadBatcher readBatcher;
    @Autowired
    private KikimrKvClient kikimrKvClient;
    @Autowired
    private KikimrKvClientSync kikimrKvClientSync;


    private long createTabletAndWaitForCreation(long ownerId, long idx) {
        if (kikimrKvClient instanceof KikimrKvClientInMem) {
            return ((KikimrKvClientInMem) kikimrKvClient).createKvTablet();
        }

        final String path = "/local/" + UUID.randomUUID();
        kikimrKvClientSync.createKvTablesWithSchemeShard(path, 1);
        long tabletId = kikimrKvClientSync.resolveKvTablesWithSchemeShard(path)[0];

        for (int i = 0; ; ++i) {
            if (i == 100) {
                throw new RuntimeException("not yet created");
            }

            if (Arrays.stream(kikimrKvClientSync.findTabletsOnLocalhost()).anyMatch(t -> t == tabletId)) {
                break;
            }

            ThreadUtils.sleep(10, TimeUnit.MILLISECONDS);
        }
        return tabletId;
    }

    @Test
    public void success() {
        long tabletId = createTabletAndWaitForCreation(123456, 10);
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);

        List<KikimrKvClient.Write> writes = IntStream.range(0, 1000)
            .mapToObj(i -> {
                return new KikimrKvClient.Write("a" + i, ("a" + i).getBytes(StandardCharsets.UTF_8),
                    MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, KikimrKvClient.Write.defaultPriority);
            })
            .collect(Collectors.toList());
        kikimrKvClientSync.writeMulti(tabletId, gen, writes);

        ArrayList<CompletableFuture<byte[]>> futures = new ArrayList<>();

        for (int i = 0; i < 1000; ++i) {
            ThreadLocalTimeout.Handle h = ThreadLocalTimeout.pushTimeout(1, TimeUnit.MINUTES);
            try {
                futures.add(readBatcher.kvReadData(tabletId, gen, new KvChunkAddress("a" + i)));
                ThreadUtils.sleep(1);
            } finally {
                h.popSafely();
            }
        }

        for (int i = 0; i < 1000; ++i) {
            Assert.assertArrayEquals(("a" + i).getBytes(StandardCharsets.UTF_8), futures.get(i).join());
        }
    }

    @Test
    public void error() throws InterruptedException {
        long tabletId = createTabletAndWaitForCreation(123456, 10);
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);

        var result = readBatcher.kvReadData(tabletId, gen, new KvChunkAddress("nonexistent"))
                .thenApply(ignore -> (Throwable) null)
                .exceptionally(e -> e)
                .join();

        assertNotNull(result);
    }

    @Test
    public void tlt() {
        long tabletId = createTabletAndWaitForCreation(123456, 10);
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);

        KvChunkAddress[] addresses = { new KvChunkAddress("does-not-matter") };
        long deadlineInstantMillis = System.currentTimeMillis() - 1000;
        CompletableFuture<byte[][]> future = readBatcher.kvReadDataMulti(tabletId, gen, addresses, deadlineInstantMillis);
        Try<byte[][]> aTry = CompletableFutures.getTry(future);
        Assert.assertTrue(aTry.getThrowable() instanceof StockpileRuntimeException);
        Assert.assertEquals(EStockpileStatusCode.DEADLINE_EXCEEDED, ((StockpileRuntimeException) aTry.getThrowable()).getStockpileStatusCode());
    }

    @Test
    public void flushQueue() {
        long tabletId = createTabletAndWaitForCreation(123456, 22);
        long gen = kikimrKvClientSync.incrementGeneration(tabletId);

        kikimrKvClientSync.writeDefault(tabletId, gen, "aabb", new byte[] { 10, 20 });

        ArrayList<CompletableFuture<byte[]>> futures = new ArrayList<>();
        for (int i = 0; i < readBatcher.inflightLimit * 2; ++i) {
            ThreadLocalTimeout.Handle h = ThreadLocalTimeout.pushTimeout(1, TimeUnit.MINUTES);
            try {
                futures.add(readBatcher.kvReadData(tabletId, gen, new KvChunkAddress("aabb")));
            } finally {
                h.popSafely();
            }
        }

        for (int i = 0; i < readBatcher.inflightLimit; ++i) {
            Assert.assertArrayEquals(new byte[] { 10, 20 }, futures.get(i).join());
        }

        for (int i = readBatcher.inflightLimit; i < futures.size(); ++i) {
            // cannot be reliably checked
            //Assert.assertFalse(futures.get(i).isDone());
        }

        readBatcher.flushQueueForTablet(tabletId).join();

        for (int i = readBatcher.inflightLimit; i < futures.size(); ++i) {
            Assert.assertArrayEquals(new byte[] { 10, 20 }, futures.get(i).join());
        }
    }
}
