package ru.yandex.stockpile.server.data.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KvAsyncIterator;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.log.StockpileLogEntry;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentSerializer;
import ru.yandex.stockpile.server.data.log.StockpileLogEntrySerialized;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;


/**
 * @author Sergey Polovko
 */
public class StockpileDaoForTests {

    private final KikimrKvClient kvClient;

    public StockpileDaoForTests(KikimrKvClient kvClient) {
        this.kvClient = kvClient;
    }

    public CompletableFuture<Tuple2<ArrayList<StockpileLogEntrySerialized>, Long>> readLogs(long tabletId, long gen) {
        ArrayList<StockpileLogEntrySerialized> list = new ArrayList<>();
        long[] totalSize = new long[1];
        return readLogsIter(tabletId, gen).readRemaining(e -> {
            list.addAll(e._1);
            totalSize[0] += e._2;
        }).thenApply(u -> Tuple2.tuple(list, totalSize[0]));
    }

    private KvAsyncIterator<Tuple2<ArrayList<StockpileLogEntrySerialized>, Long>> readLogsIter(long tabletId, long gen) {
        return kvClient.readRangeAllIter(tabletId, gen, StockpileKvNames.logRange(), 0)
            .map(l -> {
                ArrayList<StockpileLogEntrySerialized> logs = l.stream().map(e -> {
                    byte[] bytes = e.getValue();
                    return new StockpileLogEntrySerialized(LogFile.pfCurrent.parseOrThrow(e.getName()).txn0zForTest(), bytes);
                }).collect(CollectorsF.toJavaUtilArrayList());

                long logSize = l.stream().mapToLong(e -> e.getValue().length).sum();

                // free memory
                l.clear();
                l.trimToSize();

                return Tuple2.tuple(logs, logSize);
            });
    }

    public CompletableFuture<List<IndexFile>> readIndexList(long tabletId, long gen, SnapshotLevel level) {
        return kvClient.readRangeNames(tabletId, gen, StringMicroUtils.asciiPrefixToRange(level.index.currentPrefix()), 0)
            .thenApply(l -> {
                return l.stream().map(e -> {
                    return IndexFile.pfCurrent.parseOrThrow(e.getName());
                }).collect(Collectors.toList());
            });
    }

    public CompletableFuture<StockpileLogEntry> readLog(long tabletId, long gen, long txn) {
        String fileName = StockpileKvNames.shortLogName(txn);
        return kvClient.readDataSome(tabletId, gen, fileName, 0, -1, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME)
            .thenApply(bs -> {
                StockpileLogEntryContent content = StockpileLogEntryContentSerializer.S.deserializeFullVerbose(bs, fileName);
                return new StockpileLogEntry(txn, content);
            });
    }

    public CompletableFuture<SnapshotIndex> readIndex(long tabletId, long gen, SnapshotLevel level, long txn) {
        String fileName = new IndexFile(level, txn, 0, true).reconstructCurrent();
        return kvClient.readDataSome(tabletId, gen, fileName, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME)
            .thenApply(bs -> {
                if (bs.length == 0) {
                    throw new IllegalStateException("empty index file: " + fileName);
                }
                SnapshotIndexContent content = SnapshotIndexContentSerializer.S.deserializeFullVerbose(bs, fileName);
                return new SnapshotIndex(level, txn, content);
            });
    }
}
