package ru.yandex.stockpile.server.data.dao;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Throwables;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.kikimr.client.kv.KikimrKvClient.Write;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics;
import ru.yandex.stockpile.server.shard.ExceptionHandler;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class TabletReaderTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private KikimrKvClientInMem kvClient;
    private long tabletId;
    private long gen;
    private TabletReader reader;

    @Before
    public void setUp() throws Exception {
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        gen = kvClient.incrementGeneration(tabletId, 0).join();
        var registry = new MetricRegistry();
        reader = new TabletReader(
                tabletId,
                10,
                new KikimrKvClientCounting(kvClient, new KikimrKvClientMetrics(registry)),
                ForkJoinPool.commonPool(),
                registry);
    }

    @Test
    public void readOne() {
        kvClient.writeMulti(tabletId, gen, List.of(prepareWrite("file", "value")), 0).join();

        var result = reader.read(gen, new KvChunkAddress("file"), expiredAt()).join();
        assertArrayEquals("value".getBytes(StandardCharsets.UTF_8), result);
    }

    @Test
    public void readMany() {
        List<Write> writes = IntStream.range(0, 1000)
                .mapToObj(i -> prepareWrite("a" + i, ("a" + i)))
                .collect(Collectors.toList());
        kvClient.writeMulti(tabletId, gen, writes, 0).join();

        var results = IntStream.range(0, 1000)
                .mapToObj(i -> reader.read(gen, new KvChunkAddress("a" + i), expiredAt()))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
                .join();

        for (int index = 0; index < 1000; index++) {
            assertArrayEquals(("a" + index).getBytes(StandardCharsets.UTF_8), results.get(index));
        }
    }

    @Test
    public void deadlineNearExpire() {
        kvClient.writeMulti(tabletId, gen, List.of(prepareWrite("file", "value")), 0).join();

        var result = reader.read(gen, new KvChunkAddress("file"), System.currentTimeMillis() + 1)
                .thenApply(bytes -> EStockpileStatusCode.OK)
                .exceptionally(throwable -> ((StockpileRuntimeException) Throwables.getRootCause(throwable)).getStockpileStatusCode())
                .join();

        assertEquals(EStockpileStatusCode.DEADLINE_EXCEEDED, result);
    }

    @Test
    public void generationChanged() {
        kvClient.writeMulti(tabletId, gen, List.of(prepareWrite("file", "value")), 0).join();

        var result = reader.read(gen + 1, new KvChunkAddress("file"), expiredAt())
                .thenApply(bytes -> (Throwable) null)
                .exceptionally(throwable -> throwable)
                .join();

        assertTrue(ExceptionHandler.isGenerationChanged(result));
    }

    @Test
    public void readMultiChunk() {
        kvClient.writeMulti(tabletId, gen, List.of(
                prepareWrite("file.001", "one"),
                prepareWrite("file.002", "two")),
                expiredAt()).join();

        var result = reader.read(gen, new KvChunkAddress[]{new KvChunkAddress("file.001"), new KvChunkAddress("file.002")}, expiredAt()).join();
        assertEquals(2, result.length);
        assertArrayEquals("one".getBytes(StandardCharsets.UTF_8), result[0]);
        assertArrayEquals("two".getBytes(StandardCharsets.UTF_8), result[1]);
    }

    @Test
    public void failedReads() {
        kvClient.writeMulti(tabletId, gen, List.of(prepareWrite("file", "value")), 0).join();

        kvClient.throwOnRead();
        var address = new KvChunkAddress("file");
        for (int index = 0; index < 3; index++) {
            var failed = reader.read(gen, address, expiredAt())
                    .thenApply(bytes -> Boolean.FALSE)
                    .exceptionally(throwable -> Boolean.TRUE)
                    .join();
            assertTrue(failed);
        }

        kvClient.resumeReads();
        var result = reader.read(gen, address, expiredAt()).join();
        assertArrayEquals("value".getBytes(StandardCharsets.UTF_8), result);
    }

    private Write prepareWrite(String name, String value) {
        return new Write(name, value.getBytes(StandardCharsets.UTF_8), EStorageChannel.MAIN, Write.defaultPriority);
    }

    private long expiredAt() {
        return System.currentTimeMillis() + 10_000;
    }
}
