package ru.yandex.stockpile.server.data.index;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;

import com.google.common.collect.Lists;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.chunk.ChunkIndex;

import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class ChunkIndexArrayTest {
    private ChunkIndexArray array;

    @Before
    public void setUp() {
        array = new ChunkIndexArray();
    }

    @Test
    public void findMetricLastTsMillisEmpty() {
        final long localId = StockpileLocalId.random();
        assertEquals(-1, array.findMetricLastTsMillis(localId));
    }

    @Test
    public void findMetricLastTsMillisGreaterOutsize() {
        final long localId = StockpileLocalId.random();
        final long now = System.currentTimeMillis();

        array.addMetric(localId, now, 100);
        array.addMetric(localId + 1, now, 100);
        array.addMetric(localId + 2, now, 100);
        array.addMetric(localId + 3, now, 100);
        array.finishChunk();

        assertEquals(-1, array.findMetricLastTsMillis(localId - 1));
        assertEquals(-1, array.findMetricLastTsMillis(localId - 5));
        assertEquals(-1, array.findMetricLastTsMillis(localId + 5));
    }

    @Test
    public void findMetricLastTsMillis() {
        var metrics = IntStream.range(0, 1000)
            .mapToObj(ignore -> Metric.random())
            .sorted()
            .collect(toList());
        addMetrics(metrics);

        for (Metric metric : metrics) {
            checkFindMetricLastTsMillis(metric);
        }
        assertEquals(-1, array.findMetricLastTsMillis(StockpileLocalId.random()));
    }

    @Test
    public void findMetricDataManyEmptyChunks() {
        var metrics = IntStream.range(0, 100)
            .mapToObj(ignore -> Metric.random())
            .sorted()
            .collect(toList());
        var partitions = Lists.partition(metrics, 100);
        addMetricsByPartitions(partitions);
        checkFindMetric(partitions);
    }

    @Test
    public void findMetricDataManyChunks() {
        var metrics = IntStream.range(0, 1000)
            .mapToObj(ignore -> Metric.random())
            .sorted()
            .collect(toList());
        var partitions = Lists.partition(metrics, 42);
        addMetricsByPartitions(partitions);
        checkFindMetric(partitions);
    }

    @Test
    public void getChunkOne() {
        var metrics = IntStream.range(0, 100)
            .mapToObj(ignore -> Metric.random())
            .sorted()
            .collect(toList());

        var partitions = Lists.partition(metrics, 100);
        addMetricsByPartitions(partitions);
        checkGetChunk(partitions);
        array = new ChunkIndexArray(array.chunks());
        checkGetChunk(partitions);
    }

    @Test
    public void getChunksCount() {
        assertEquals(0, array.getChunksCount());
        assertEquals(0, array.chunks().length);

        array.addMetric(1, 1000, 42);
        array.addMetric(2, 2000, 43);
        array.finishChunk();
        assertEquals(1, array.getChunksCount());
        assertEquals(1, array.chunks().length);

        array.finishChunk();
        assertEquals(1, array.getChunksCount());
        assertEquals(1, array.chunks().length);

        array.addMetric(3, 3000, 44);
        array.addMetric(4, 4000, 45);
        array.finishChunk();
        assertEquals(2, array.getChunksCount());
        assertEquals(2, array.chunks().length);

        array = new ChunkIndexArray(array.chunks());
        assertEquals(2, array.getChunksCount());
        assertEquals(2, array.chunks().length);
    }

    @Test
    public void getChunkMany() {
        var metrics = IntStream.range(0, 1000)
            .mapToObj(ignore -> Metric.random())
            .sorted()
            .collect(toList());

        var partitions = Lists.partition(metrics, 42);
        addMetricsByPartitions(partitions);
        checkGetChunk(partitions);
        array = new ChunkIndexArray(array.chunks());
        checkGetChunk(partitions);
    }

    @Test
    public void shrinkToFit() {
        assertEquals(0, array.getMetricCapacity());
        assertEquals(0, array.getMetricCount());

        var metrics = IntStream.range(0, 1000)
                .mapToObj(ignore -> Metric.random())
                .sorted()
                .collect(toList());

        var partitions = Lists.partition(metrics, 42);
        addMetricsByPartitions(partitions);
        assertEquals(metrics.size(), array.getMetricCount());
        assertNotEquals(metrics.size(), array.getMetricCapacity());

        array.shrinkToFit();
        assertEquals(metrics.size(), array.getMetricCount());
        assertEquals(metrics.size(), array.getMetricCapacity());

        checkGetChunk(partitions);
        checkFindMetric(partitions);
        array = new ChunkIndexArray(array.chunks());
        checkGetChunk(partitions);
        checkFindMetric(partitions);
    }

    @Test
    public void oneMetricDiskSize() {
        var metric = Metric.random();

        addMetrics(List.of(metric));
        assertEquals(metric.size, array.countChunkDiskSize());
    }

    private void checkFindMetricLastTsMillis(Metric metric) {
        assertEquals(metric.lastTsMillis, array.findMetricLastTsMillis(metric.localId));
    }

    private void addMetricsByPartitions(List<List<Metric>> partitions) {
        for (var partition : partitions) {
            addMetrics(partition);
        }
    }

    private void checkFindMetric(List<List<Metric>> partitions) {
        for (int chunkNo = 0; chunkNo < partitions.size(); chunkNo++) {
            var partition = partitions.get(chunkNo);
            int offset = 0;
            for (var metric : partition) {
                var result = array.findMetric(metric.localId);
                assertNotNull(result);
                assertEquals(chunkNo, result.getChunkNo());
                assertEquals(metric.localId, result.getMetricId());
                assertEquals(metric.lastTsMillis, result.getLastTsMillis());
                assertEquals(offset, result.getOffset());
                assertEquals(metric.size, result.getSize());
                offset += metric.size;
            }
        }
        assertNull(array.findMetricData(StockpileLocalId.random()));
    }

    private void checkGetChunk(List<List<Metric>> partitions) {
        for (int chunkNo = 0; chunkNo < partitions.size(); chunkNo++) {
            var partition = partitions.get(chunkNo);

            ChunkIndex chunk = array.getChunk(chunkNo);
            for (int index = 0; index < partition.size(); index++) {
                var metric = partition.get(index);

                assertEquals(metric.localId, chunk.getLocalIdsSortedArray()[index]);
                assertEquals(metric.lastTsMillis, chunk.getLastTssMillisArray()[index]);
                assertEquals(metric.size, chunk.getSize(index));
            }
        }
    }

    private void addMetrics(List<Metric> metrics) {
        for (var metric : metrics) {
            array.addMetric(metric.localId, metric.lastTsMillis, metric.size);
        }
        array.finishChunk();
    }

    private static class Metric implements Comparable<Metric> {
        private long localId;
        private long lastTsMillis;
        private int size;

        public static Metric random() {
            ThreadLocalRandom random = ThreadLocalRandom.current();

            var metric = new Metric();
            metric.localId = StockpileLocalId.random(random);
            metric.lastTsMillis = random.nextLong();
            metric.size = random.nextInt(10, 10_000);
            return metric;
        }

        @Override
        public int compareTo(@Nonnull Metric o) {
            return StockpileLocalId.compare(localId, o.localId);
        }

        @Override
        public String toString() {
            return StockpileLocalId.toString(localId);
        }
    }
}
