package ru.yandex.stockpile.server.data.index.stats;

import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Collectors;

import com.google.common.io.Resources;
import org.junit.Test;

import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.HeapStockpileSerializer;
import ru.yandex.solomon.codec.serializer.SelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.server.data.index.Protobuf5SerializerFactorySolomon;
import ru.yandex.stockpile.server.data.index.SelfContainedSerializerProtobuf5LengthDelimited;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class IndexStatsProjectTest {

    @Test
    public void serializeDeserialize() {
        IndexStatsProject stats = new IndexStatsProject();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        one.setOwnerShardId(42);
        one.addRecord(randomPoint(MetricType.DGAUGE));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.IGAUGE);
        two.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        two.setOwnerShardId(42);
        two.addRecord(randomPoint(MetricType.IGAUGE));
        two.addRecord(randomPoint(MetricType.IGAUGE));

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setType(MetricType.LOG_HISTOGRAM);
        tree.setOwnerProjectIdEnum(EProjectId.GOLOVAN);
        tree.addRecord(randomPoint(MetricType.LOG_HISTOGRAM));

        stats.add(one);
        stats.add(two);
        stats.add(tree);

        String expected = supplyToString(stats);

        byte[] bytes = serialize(stats);
        var result = deserialize(bytes);

        String actual = supplyToString(result);
        assertEquals(expected, actual);
        close(one, two, tree);
    }

    @Test
    public void stableTest() throws IOException {
        String expected = loadTextForm("IndexStatsProjectTest.1.txt");
        byte[] bytes = Resources.toByteArray(getClass().getResource("IndexStatsProjectTest.1.bin"));

        IndexStatsProject stats = deserialize(bytes);
        String actual = supplyToString(stats);
        assertEquals(expected, actual);
    }

    private void genBinary(Path path, String name, IndexStatsProject stats) throws IOException {
        String str = supplyToString(stats);
        Files.write(path.resolve(name + ".txt"), str.getBytes());
        byte[] bytes = serialize(stats);
        Files.write(path.resolve(name + ".bin"), bytes);
    }

    private String supplyToString(IndexStatsProject stats) {
        StringWriter out = new StringWriter();
        try (MetricTextEncoder e = new MetricTextEncoder(out, true)) {
            stats.supply(0, e);
        }

        String result = out.toString()
            .lines()
            .sorted()
            .filter(line -> !line.isBlank())
            .collect(Collectors.joining("\n"));
        System.out.println(result);
        return result;
    }

    private String loadTextForm(String file) throws IOException {
        return Resources.toString(getClass().getResource(file), StandardCharsets.UTF_8)
            .lines()
            .sorted()
            .filter(line -> !line.isBlank())
            .collect(Collectors.joining("\n"));
    }

    private byte[] serialize(IndexStatsProject stats) {
        var content = new HeapStockpileSerializer();
        var serializer = getSerializer();
        serializer.serializeWithLength(stats, content);
        return content.build();
    }

    private IndexStatsProject deserialize(byte[] bytes) {
        return getSerializer().deserializeWithLength(new StockpileDeserializer(bytes));
    }

    private SelfContainedSerializer<IndexStatsProject> getSerializer() {
        return new SelfContainedSerializerProtobuf5LengthDelimited<>(new Protobuf5SerializerFactorySolomon().forClass(IndexStatsProject.class));
    }

}
