package ru.yandex.stockpile.server.data.log;

import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;

import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongMaps;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.DeletedShardSet;

import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assume.assumeThat;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class StockpileLogEntryContentSerializerTest {
    @Parameterized.Parameter
    public StockpileFormat format;

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return StockpileFormat.values();
    }

    @Test
    public void serializeDeserializeMetrics() {
        var expected = new StockpileLogEntryContent(randomMetrics(), new DeletedShardSet(), Int2LongMaps.EMPTY_MAP);
        var actual = deserialize(serialize(expected));
        assertEquals(expected.getDataByMetricId(), actual.getDataByMetricId());
    }

    @Test
    public void serializeDeserialize() {
        assumeThat(format.getFormat(), greaterThanOrEqualTo(StockpileFormat.IDEMPOTENT_WRITE_38.getFormat()));

        var expected = new StockpileLogEntryContent(randomMetrics(), new DeletedShardSet(), randomProducerSeqNo());
        var actual = deserialize(serialize(expected));
        assertEquals(expected.getDataByMetricId(), actual.getDataByMetricId());
        assertEquals(expected.getProducerSeqNoById(), actual.getProducerSeqNoById());
    }

    @Test
    public void compatibleFormatWithImmutableMetrics() {
        var expected = new StockpileLogEntryContent(randomMetrics(), new DeletedShardSet(), Int2LongMaps.EMPTY_MAP);
        var mutable = serialize(expected);
        var immutable = serialize(toImmutable(expected));

        for (var bytes : Arrays.asList(immutable, mutable)) {
            {
                var result = deserializeImmutable(bytes);
                assertEquals(expected.getProducerSeqNoById(), result.getProducerSeqNoById());
                assertEquals(expected.getDataByMetricId(), toMutable(result).getDataByMetricId());
            }
            {
                var result = deserialize(bytes);
                assertEquals(expected.getProducerSeqNoById(), result.getProducerSeqNoById());
                assertEquals(expected.getDataByMetricId(), result.getDataByMetricId());
            }
        }
    }

    @Test
    public void compatibleFormatWithImmutable() {
        assumeThat(format.getFormat(), greaterThanOrEqualTo(StockpileFormat.IDEMPOTENT_WRITE_38.getFormat()));

        var expected = new StockpileLogEntryContent(randomMetrics(), new DeletedShardSet(), randomProducerSeqNo());
        var mutable = serialize(expected);
        var immutable = serialize(toImmutable(expected));

        for (var bytes : Arrays.asList(immutable, mutable)) {
            {
                var result = deserializeImmutable(bytes);
                assertEquals(expected.getProducerSeqNoById(), result.getProducerSeqNoById());
                assertEquals(expected.getDataByMetricId(), toMutable(result).getDataByMetricId());
            }
            {
                var result = deserialize(bytes);
                assertEquals(expected.getProducerSeqNoById(), result.getProducerSeqNoById());
                assertEquals(expected.getDataByMetricId(), result.getDataByMetricId());
            }
        }
    }

    @Test
    public void serializeDeserializeDeletedShards() {
        assumeThat(format.getFormat(), greaterThanOrEqualTo(StockpileFormat.DELETED_SHARDS_39.getFormat()));

        var expected = new StockpileLogEntryContent(randomMetrics(), randomDeletedShards(), randomProducerSeqNo());
        var actual = deserialize(serialize(expected));
        assertEquals(expected.getDataByMetricId(), actual.getDataByMetricId());
        assertEquals(expected.getProducerSeqNoById(), actual.getProducerSeqNoById());
        assertEquals(expected.getDeletedShards(), actual.getDeletedShards());
    }

    private StockpileLogEntryContentImmutable toImmutable(StockpileLogEntryContent content) {
        var archives = new Long2ObjectOpenHashMap<MetricArchiveImmutable>(content.getDataByMetricId().size());
        for (var entry : content.getDataByMetricId().long2ObjectEntrySet()) {
            archives.put(entry.getLongKey(), entry.getValue().toImmutable());
        }
        return new StockpileLogEntryContentImmutable(archives, content.getDeletedShards(), content.getProducerSeqNoById());
    }

    private StockpileLogEntryContent toMutable(StockpileLogEntryContentImmutable content) {
        var archives = new Long2ObjectOpenHashMap<MetricArchiveMutable>(content.getDataByMetricId().size());
        for (var entry : content.getDataByMetricId().long2ObjectEntrySet()) {
            archives.put(entry.getLongKey(), entry.getValue().toMutable());
        }
        return new StockpileLogEntryContent(archives, content.getDeletedShards(), content.getProducerSeqNoById());
    }

    private byte[] serialize(StockpileLogEntryContent log) {
        return new StockpileLogEntryContentSerializer(format).serializeToBytes(log);
    }

    public byte[] serialize(StockpileLogEntryContentImmutable log) {
        return new StockpileLogEntryContentImmutableSerializer(format).serializeToBytes(log);
    }

    private StockpileLogEntryContent deserialize(byte[] serialized) {
        return StockpileLogEntryContentSerializer.S.deserializeFull(serialized);
    }

    private StockpileLogEntryContentImmutable deserializeImmutable(byte[] serialized) {
        return StockpileLogEntryContentImmutableSerializer.S.deserializeFull(serialized);
    }

    private Long2ObjectOpenHashMap<MetricArchiveMutable> randomMetrics() {
        var count = ThreadLocalRandom.current().nextInt(1, 100);
        var result = new Long2ObjectOpenHashMap<MetricArchiveMutable>(count);
        for (int index = 0; index < count; index++) {
            long localId = StockpileLocalId.random();
            var archive = randomArchive();
            result.put(localId, archive);
        }
        return result;
    }

    private MetricArchiveMutable randomArchive() {
        var random = ThreadLocalRandom.current();

        var archive = new MetricArchiveMutable();
        archive.setOwnerShardId(random.nextInt());
        archive.setType(randomMetricType());

        int mask = randomMask(archive.getType());
        var point = RecyclableAggrPoint.newInstance();
        for (int index = 0; index < 5; index++) {
            archive.addRecord(randomPoint(point, mask, random));
        }
        point.recycle();
        archive.sortAndMerge();
        return archive;
    }

    private Int2LongMap randomProducerSeqNo() {
        var random = ThreadLocalRandom.current();
        int count = random.nextInt(0,10_000);
        Int2LongMap result = new Int2LongOpenHashMap(count);
        for (int index = 0; index < count; index++) {
            int producerId = random.nextInt();
            long producerSeqNo = random.nextLong();
            result.put(producerId, producerSeqNo);
        }
        return result;
    }

    private DeletedShardSet randomDeletedShards() {
        var random = ThreadLocalRandom.current();
        var count = random.nextInt(0, 10_000);
        var result = new DeletedShardSet(count);
        for (int index = 0; index < count; index++) {
            int projectId = random.nextInt(3);
            int shardId = random.nextInt();
            result.add(projectId, shardId);
        }
        return result;
    }
}
