package ru.yandex.stockpile.server.data.names;

import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.kikimr.util.NameRange;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Stepan Koltsov
 */
public class StockpileKvNamesTest {

    private static BigFilePart[] bigFileParts() {
        return new BigFilePart[] {
            new BigFilePart(0, true),
            new BigFilePart(1, true),
            new BigFilePart(9, true),
            new BigFilePart(0, false),
            new BigFilePart(1, false),
            new BigFilePart(9, false),
        };
    }

    @Test
    public void logsToRangeExcludeTo() {
        NameRange nameRange = StockpileKvNames.logsToRange(5000, false);

        for (BigFilePart part : bigFileParts()) {
            Assert.assertTrue(nameRange.includes(new LogFile(2000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        for (BigFilePart part : bigFileParts()) {
            Assert.assertFalse(nameRange.includes(new LogFile(8000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        for (BigFilePart part : bigFileParts()) {
            Assert.assertFalse(nameRange.includes(new LogFile(5000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        Assert.assertFalse(nameRange.includes(new IndexFile(SnapshotLevel.TWO_HOURS, 10000, 2, false).reconstructCurrent()));
    }

    @Test
    public void logsToRangeIncludeTo() {
        NameRange nameRange = StockpileKvNames.logsToRange(5000, true);

        for (BigFilePart part : bigFileParts()) {
            Assert.assertTrue(nameRange.includes(new LogFile(2000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        for (BigFilePart part : bigFileParts()) {
            Assert.assertFalse(nameRange.includes(new LogFile(8000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        for (BigFilePart part : bigFileParts()) {
            Assert.assertTrue(nameRange.includes(new LogFile(5000, part.getNo(), part.isLast()).reconstructCurrent()));
        }

        Assert.assertFalse(nameRange.includes(new IndexFile(SnapshotLevel.TWO_HOURS, 10000, 2, false).reconstructCurrent()));
    }

    @Test
    public void logNameRangeInclusive() {
        NameRange range = StockpileKvNames.logNameRangeInclusive(24585504, 24585604 - 1);
        System.out.println(range);
        // last txn only one chunk
        assertTrue(range.includes("c.l.00000000024585603.00000z"));

        // last txn 3 chunks
        assertTrue(range.includes("c.l.00000000024585603.00001y"));
        assertTrue(range.includes("c.l.00000000024585603.00002y"));
        assertTrue(range.includes("c.l.00000000024585603.00003z"));

        // first tx only one chunk
        assertTrue(range.includes("c.l.00000000024585504.00000z"));

        // first txn 3 chunks
        assertTrue(range.includes("c.l.00000000024585504.00001y"));
        assertTrue(range.includes("c.l.00000000024585504.00002y"));
        assertTrue(range.includes("c.l.00000000024585504.00003z"));

        assertTrue(range.isBeginInclusive());
        assertEquals("c.l.00000000024585504", range.getBegin());

        assertTrue(range.isEndInclusive());
        assertEquals("c.l.00000000024585603.99999z", range.getEnd());
    }

    @Test
    public void snapshotFilesLevel() {
        {
            var range = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.TWO_HOURS, 1));
            assertTrue(isInclude(range, "c.s0.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.i0.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.c0.00000000000000001.00034y"));
        }
        {
            var range = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.DAILY, 1));
            assertTrue(isInclude(range, "c.s1.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.i1.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.c1.00000000000000001.00034y"));
        }
        {
            var range = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.ETERNITY, 1));
            assertTrue(isInclude(range, "c.s2.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.i2.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.c2.00000000000000001.00034y"));
        }
    }

    @Test
    public void snapshotFilesTxn() {
        {
            var range = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.TWO_HOURS, 1));
            assertTrue(isInclude(range, "c.s0.00000000000000001.00034y"));
            assertTrue(isInclude(range, "c.s0.00000000000000001.00035z"));
        }
        {
            var range = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.TWO_HOURS, 123));
            assertTrue(isInclude(range, "c.s0.00000000000000123.00034y"));
            assertTrue(isInclude(range, "c.s0.00000000000000123.00035z"));
        }
    }

    @Test
    public void snapshotFilesNoMatch() {
        var ranges = StockpileKvNames.currentSnapshot(new SnapshotAddress(SnapshotLevel.TWO_HOURS, 1234567));
        assertFalse(isInclude(ranges, "s0.00000000001234567.00034y"));
        assertFalse(isInclude(ranges, "c.d0.00000000001234567.00034y"));
        assertFalse(isInclude(ranges, "c.s1.00000000001234567.00034y"));
        assertFalse(isInclude(ranges, "c.s2.00000000001234567.00034y"));
        assertFalse(isInclude(ranges, "c.s0.00000000001234566.00034y"));
        assertFalse(isInclude(ranges, "c.s0.00000000001234568.00034y"));
        assertFalse(isInclude(ranges, "c.c0.00000000001234568.00034y"));
        assertFalse(isInclude(ranges, "c.s0.junk"));
        assertFalse(isInclude(ranges, "c.c0.junk"));
        assertFalse(isInclude(ranges, "c.i0.junk"));
    }

    private boolean isInclude(List<NameRange> ranges, String file) {
        for (var range : ranges) {
            if (range.includes(file)) {
                return true;
            }
        }

        return false;
    }
}
