package ru.yandex.stockpile.server.shard;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.DataRangeGlobal;
import ru.yandex.stockpile.server.data.index.ChunkIndexArray;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexProperties;
import ru.yandex.stockpile.server.shard.stat.LevelSizeAndCount;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;


/**
 * @author Vladimir Gordiychuk
 */
public class AllIndexesTest {
    private long txn;
    private AllIndexes indexes;

    @Before
    public void setUp() {
        txn = 0;
        indexes = new AllIndexes("", List.of(), List.of(), List.of());
    }

    @Test
    public void empty() {
        var result = indexArray();
        assertArrayEquals(new SnapshotIndexWithStats[0], result);
        assertEquals(SnapshotTs.SpecialTs.NEVER.value, indexes.latestSnapshotTime(SnapshotLevel.TWO_HOURS));
        assertEquals(SnapshotTs.SpecialTs.NEVER.value, indexes.latestSnapshotTime(SnapshotLevel.DAILY));
        assertEquals(SnapshotTs.SpecialTs.NEVER.value, indexes.latestSnapshotTime(SnapshotLevel.ETERNITY));
    }

    @Test
    public void twoHours() {
        var expected = IntStream.range(0, 3)
            .mapToObj(ignore -> index(SnapshotLevel.TWO_HOURS))
            .toArray(SnapshotIndexWithStats[]::new);

        for (var snapshot : expected) {
            indexes.addSnapshot(snapshot);
            indexes.updateLatestSnapshotTime(SnapshotLevel.TWO_HOURS, snapshot.getIndex().getContent().getTsMillis());
        }
        var result = indexArray();
        assertArrayEquals(expected, result);

        long latest = indexes.latestSnapshotTime(SnapshotLevel.TWO_HOURS);
        assertEquals(expected[2].getIndex().getContent().getTsMillis(), latest);
    }

    @Test
    public void daily() {
        var twoHours = index(SnapshotLevel.TWO_HOURS);
        var daily = index(SnapshotLevel.DAILY);

        indexes.addSnapshot(twoHours);
        assertArrayEquals(array(twoHours), indexArray());

        indexes.removeSnapshot(twoHours);
        indexes.addSnapshot(daily);

        assertArrayEquals(array(daily), indexArray());

        var anotherTwoHours = index(SnapshotLevel.TWO_HOURS);
        indexes.addSnapshot(anotherTwoHours);
        assertArrayEquals(array(daily, anotherTwoHours), indexArray());
    }

    @Test
    public void twoHoursBeforeDaily() {
        var one = index(SnapshotLevel.TWO_HOURS);
        var two = index(SnapshotLevel.TWO_HOURS);
        var daily = index(SnapshotLevel.DAILY);
        var three = index(SnapshotLevel.TWO_HOURS);

        indexes.addSnapshot(one);
        indexes.addSnapshot(two);
        // daily started but not completed yet
        indexes.addSnapshot(three);

        assertArrayEquals(array(one, two, three), indexArray());

        indexes.removeSnapshot(one);
        indexes.removeSnapshot(two);
        indexes.addSnapshot(daily);

        assertArrayEquals(array(daily, three), indexArray());
    }

    @Test
    public void levelOrder() {
        var daily = index(SnapshotLevel.DAILY);
        var hours = index(SnapshotLevel.TWO_HOURS);
        var eternity = index(SnapshotLevel.ETERNITY);

        indexes.addSnapshot(daily);
        indexes.addSnapshot(hours);
        indexes.addSnapshot(eternity);

        assertArrayEquals(array(eternity, daily, hours), indexArray());
    }

    @Test
    public void transactionOrder() {
        var one = index(SnapshotLevel.DAILY);
        var two = index(SnapshotLevel.DAILY);
        var three = index(SnapshotLevel.DAILY);
        var four = index(SnapshotLevel.DAILY);

        indexes.addSnapshot(one);
        indexes.addSnapshot(two);

        // was added patch before previous merged
        indexes.addSnapshot(four);
        assertArrayEquals(array(one, two, four), indexArray());

        // previous merged now combine one and two should be before four
        indexes.removeSnapshot(one);
        indexes.removeSnapshot(two);
        indexes.addSnapshot(three);

        assertArrayEquals(array(three, four), indexArray());
    }

    @Test
    public void rangeRead() {
        long localId = StockpileLocalId.random();
        long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(10);
        long ts1 = ts0 + TimeUnit.DAYS.toMillis(1);
        long ts2 = ts0 + TimeUnit.DAYS.toMillis(2);
        long ts3 = ts0 + TimeUnit.DAYS.toMillis(5);
        long ts4 = ts0 + TimeUnit.DAYS.toMillis(8);

        var eternity1 = index(SnapshotLevel.ETERNITY);
        var eternity2 = index(SnapshotLevel.ETERNITY, localId, ts0, 1000);
        var daily1 = index(SnapshotLevel.DAILY, localId, ts1, 100);
        var daily2 = index(SnapshotLevel.DAILY, localId, ts2, 10);
        var twoHours1 = index(SnapshotLevel.TWO_HOURS, localId, ts3, 1);
        var twoHours2 = index(SnapshotLevel.TWO_HOURS);

        indexes.addSnapshot(eternity1);
        indexes.addSnapshot(eternity2);
        indexes.addSnapshot(daily1);
        indexes.addSnapshot(daily2);
        indexes.addSnapshot(twoHours1);
        indexes.addSnapshot(twoHours2);

        {
            var result = indexes.rangeRequestsForMetric(StockpileLocalId.random(), ts2);
            assertArrayEquals(new DataRangeGlobal[0], result.getRanges());
        }

        {
            var result = indexes.rangeRequestsForMetric(localId, ts4);
            assertEquals(ts3 + 1, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[0], result.getRanges());
        }

        {
            var result = indexes.rangeRequestsForMetric(localId, ts3 + 1);
            assertEquals(ts3 + 1, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[0], result.getRanges());
        }

        {
            var result = indexes.rangeRequestsForMetric(localId, ts3 - 1);
            assertEquals(ts2 + 1, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[]{range(twoHours1, localId)}, result.getRanges());
        }

        {
            var result = indexes.rangeRequestsForMetric(localId, ts2);
            assertEquals(ts1 + 1, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[]{
                range(daily2, localId),
                range(twoHours1, localId),
            }, result.getRanges());
        }

        {
            var result = indexes.rangeRequestsForMetric(localId, ts0 - 1000);
            assertEquals(0, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[]{
                range(eternity2, localId),
                range(daily1, localId),
                range(daily2, localId),
                range(twoHours1, localId),
            }, result.getRanges());
        }
    }

    @Test
    public void rangeReadWhenHeadersOnlyInLastEternity() {
        long localId = StockpileLocalId.random();
        long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(10);

        var eternity1 = index(SnapshotLevel.ETERNITY, localId, ts0, 577);
        // headers only, as a result of merge from daily to eternity
        var eternity2 = index(SnapshotLevel.ETERNITY, localId, 0, 12);

        var daily1 = index(SnapshotLevel.DAILY);

        indexes.addSnapshot(eternity1);
        indexes.addSnapshot(eternity2);
        indexes.addSnapshot(daily1);

        // before point
        {
            var result = indexes.rangeRequestsForMetric(localId, ts0 - TimeUnit.DAYS.toMillis(1));
            assertEquals(0, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[]{range(eternity1, localId), range(eternity2, localId)}, result.getRanges());
        }
        // from point
        {
            var result = indexes.rangeRequestsForMetric(localId, ts0);
            assertEquals(0, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[]{range(eternity1, localId), range(eternity2, localId)}, result.getRanges());
        }
        // after point
        {
            var result = indexes.rangeRequestsForMetric(localId, ts0 + TimeUnit.DAYS.toMillis(1));
            assertEquals(ts0 + 1, result.getFromTimeMillis());
            assertArrayEquals(new DataRangeGlobal[0], result.getRanges());
        }
    }

    private SnapshotIndexWithStats[] array(SnapshotIndexWithStats... array) {
        return array;
    }

    private SnapshotIndexWithStats[] indexArray() {
        return indexes.streamWithStats()
            .toArray(SnapshotIndexWithStats[]::new);
    }

    private SnapshotIndexWithStats index(SnapshotLevel level) {
        SnapshotIndex index = new SnapshotIndex(level, ++txn, randomContent());
        return new SnapshotIndexWithStats(index, LevelSizeAndCount.zero);
    }

    private SnapshotIndexContent randomContent() {
        ChunkIndexArray chunks = new ChunkIndexArray();
        var random = ThreadLocalRandom.current();
        IntStream.range(0, 100)
            .mapToObj(value -> StockpileLocalId.random())
            .sorted(StockpileLocalId::compare)
            .forEachOrdered(localId -> {
                chunks.addMetric(localId, random.nextLong(), random.nextInt());
            });
        chunks.finishChunk();

        SnapshotIndexProperties properties = new SnapshotIndexProperties()
            .setCreatedAt(System.currentTimeMillis())
            .setRecordCount(random.nextLong(1_000))
            .setMetricCount(10);

        return new SnapshotIndexContent(StockpileFormat.CURRENT, properties, chunks);
    }

    private SnapshotIndexWithStats index(SnapshotLevel level, long localId, long lastTsMillis, int bytes) {
        SnapshotIndex index = new SnapshotIndex(level, ++txn, content(localId, lastTsMillis, bytes));
        return new SnapshotIndexWithStats(index, LevelSizeAndCount.zero);
    }

    private SnapshotIndexContent content(long localId, long lastTsMillis, int bytes) {
        var random = ThreadLocalRandom.current();

        ChunkIndexArray chunks = new ChunkIndexArray();
        chunks.addMetric(localId, lastTsMillis, bytes);
        chunks.finishChunk();

        SnapshotIndexProperties properties = new SnapshotIndexProperties()
            .setCreatedAt(System.currentTimeMillis())
            .setRecordCount(random.nextLong(1_000))
            .setMetricCount(1);

        return new SnapshotIndexContent(StockpileFormat.CURRENT, properties, chunks);
    }


    private DataRangeGlobal range(SnapshotIndexWithStats stat, long localId) {
        var index = stat.getIndex();
        var entry = index.getContent().findMetric(localId);
        assertNotNull(entry);
        return new DataRangeGlobal(index.getLevel(), index.getTxn(), index.getContent().getFormat(), entry.getDataRange());
    }
}
