package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterable;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.server.shard.ArchiveCombiner.CombineResult;
import ru.yandex.stockpile.server.shard.merge.Utils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static ru.yandex.solomon.model.point.AggrPoints.dhistogram;
import static ru.yandex.solomon.model.point.AggrPoints.lpoint;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.MetricsMatcher.assertEqualTo;

/**
 * @author Vladimir Gordiychuk
 */
public class ArchiveCombinerTest {
    @Test
    public void combineOne() {
        MetricArchiveMutable source = new MetricArchiveMutable();
        source.setOwnerShardId(42);
        source.setType(MetricType.DGAUGE);
        source.addRecord(point("2018-08-20T09:05:34Z", 123d));

        MetricArchiveMutable result = combine(source);
        assertEqualArchives(source, result);
        close(source, result);
    }

    @Test
    public void overrideByRightArchive() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(42);
        one.setType(MetricType.DGAUGE);
        one.addRecord(point("2018-08-20T09:05:30Z", 1d));
        one.addRecord(point("2018-08-20T09:05:34Z", 123d));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.DGAUGE);
        two.addRecord(point("2018-08-20T09:05:34Z", 333d));
        two.addRecord(point("2018-08-20T09:05:40Z", 435d));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerShardId(42);
        expected.setType(MetricType.DGAUGE);
        expected.addRecord(point("2018-08-20T09:05:30Z", 1d));
        expected.addRecord(point("2018-08-20T09:05:34Z", 333d));
        expected.addRecord(point("2018-08-20T09:05:40Z", 435d));

        MetricArchiveMutable result = combine(one, two);
        assertEqualArchives(expected, result);
        close(one, two, expected, result);
    }

    @Test
    public void correctMergeFirstTsMillis() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(0);
        one.setType(MetricType.COUNTER);
        one.addRecord(lpoint("2018-08-20T09:05:30Z", 1));
        one.addRecord(lpoint("2018-08-20T09:05:34Z", 123));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.DGAUGE);

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setType(MetricType.COUNTER);
        tree.addRecord(lpoint("2018-08-20T09:05:34Z", 456));

        var combiner = new ArchiveCombiner(1, 42L);
        combiner.add(one);
        combiner.add(two);
        combiner.add(tree);

        var result = combiner.combine();
        assertEquals(result.getHeader(), two.header().withKind(MetricType.COUNTER));
        var frame = result.getItemIterator().next();
        assertNotNull(frame);
        assertEquals(timeToMillis("2018-08-20T09:05:30Z"), frame.getFirstTsMillis());
        assertEquals(timeToMillis("2018-08-20T09:05:34Z"), frame.getLastTsMillis());
        assertEquals(
            AggrGraphDataArrayList.of(
                lpoint("2018-08-20T09:05:30Z", 1),
                lpoint("2018-08-20T09:05:34Z", 456)),
            AggrGraphDataArrayList.of(frame.iterator()));
        close(one, two, tree);
    }

    @Test
    public void avoidChangeKindToUnknown() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(0);
        one.setType(MetricType.COUNTER);
        one.addRecord(lpoint("2018-08-20T09:05:30Z", 1));
        one.addRecord(lpoint("2018-08-20T09:05:34Z", 123));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.METRIC_TYPE_UNSPECIFIED);

        var combiner = new ArchiveCombiner(1, 42L);
        combiner.add(one);
        combiner.add(two);

        var result = combiner.combine();
        assertEquals(result.getHeader(), two.header().withKind(MetricType.COUNTER));
        var frame = result.getItemIterator().next();
        assertNotNull(frame);
        assertEquals(timeToMillis("2018-08-20T09:05:30Z"), frame.getFirstTsMillis());
        assertEquals(timeToMillis("2018-08-20T09:05:34Z"), frame.getLastTsMillis());
        assertEquals(one.toAggrGraphDataArrayList(), AggrGraphDataArrayList.of(frame.iterator()));
        close(one);
    }

    @Test
    public void combineDeleteBefore() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(42);
        one.setType(MetricType.DGAUGE);
        one.addRecord(point("2018-08-20T09:04:30Z", 1d));
        one.addRecord(point("2018-08-20T09:04:34Z", 2d));
        one.addRecord(point("2018-08-20T09:05:35Z", 3d));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.DGAUGE);
        two.setDeleteBefore(timeToMillis("2018-08-20T09:05:30Z"));
        two.addRecord(point("2018-08-20T09:05:40Z", 4d));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerShardId(42);
        expected.setType(MetricType.DGAUGE);
        expected.setDeleteBefore(timeToMillis("2018-08-20T09:05:30Z"));
        expected.addRecord(point("2018-08-20T09:05:35Z", 3d));
        expected.addRecord(point("2018-08-20T09:05:40Z", 4d));

        MetricArchiveMutable result = combine(one, two);
        assertEqualArchives(expected, result);
        close(one, two, expected, result);
    }

    @Test
    public void replacePointsByDeleteBefore() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(42);
        one.setType(MetricType.DGAUGE);
        one.addRecord(point("2018-08-20T09:04:30Z", 1d));
        one.addRecord(point("2018-08-20T09:04:34Z", 2d));
        one.addRecord(point("2018-08-20T09:04:35Z", 3d));

        // delete previous points and add new with new grid
        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.DGAUGE);
        two.setDeleteBefore(timeToMillis("2018-08-20T09:05:00Z"));
        two.addRecord(point("2018-08-20T09:04:00Z", 4d));

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setOwnerShardId(42);
        tree.setType(MetricType.DGAUGE);
        tree.addRecord(point("2018-08-20T09:05:00Z", 5d));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerShardId(42);
        expected.setType(MetricType.DGAUGE);
        expected.setDeleteBefore(timeToMillis("2018-08-20T09:05:00Z"));
        expected.addRecord(point("2018-08-20T09:04:00Z", 4d));
        expected.addRecord(point("2018-08-20T09:05:00Z", 5d));

        MetricArchiveMutable result = combine(one, two, tree);
        assertEqualArchives(expected, result);
        close(one, two, tree, expected, result);
    }

    @Test
    public void deleteBeforeAllAndAppend() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(42);
        one.setType(MetricType.DGAUGE);
        one.addRecord(point("2018-08-20T09:04:30Z", 1d));
        one.addRecord(point("2018-08-20T09:04:34Z", 2d));
        one.addRecord(point("2018-08-20T09:04:35Z", 3d));

        // delete previous points and add new with new grid
        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setType(MetricType.DGAUGE);
        two.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        two.addRecord(point("2018-08-20T09:04:00Z", 4d));

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setOwnerShardId(42);
        tree.setType(MetricType.DGAUGE);
        tree.addRecord(point("2018-08-20T09:05:00Z", 5d));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setOwnerShardId(42);
        expected.setType(MetricType.DGAUGE);
        expected.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        expected.addRecord(point("2018-08-20T09:04:00Z", 4d));
        expected.addRecord(point("2018-08-20T09:05:00Z", 5d));

        MetricArchiveMutable result = combine(one, two, tree);
        assertEqualArchives(expected, result);
        close(one, two, tree, expected, result);
    }

    @Test
    public void combineGaugeAndRateArchives() {
        final long ts0 = timeToMillis("2018-08-07T14:36:00Z");
        final long step = 10_000;

        MetricArchiveMutable oneGauge = new MetricArchiveMutable();
        oneGauge.setType(MetricType.DGAUGE);
        oneGauge.addAll(AggrGraphDataArrayList.of(
                point(ts0, 0),
                point(ts0 + step, 70),
                point(ts0 + step * 2, 20),
                point(ts0 + step * 3, 50)));

        MetricArchiveMutable twoGauge = new MetricArchiveMutable();
        twoGauge.setType(MetricType.DGAUGE);
        twoGauge.addAll(AggrGraphDataArrayList.of(
                point(ts0 + step * 4, 20),
                point(ts0 + step * 5, 25),
                point(ts0 + step * 6, 22),
                point(ts0 + step * 7, 30)));

        MetricArchiveMutable rate = new MetricArchiveMutable();
        rate.setType(MetricType.RATE);
        rate.addAll(AggrGraphDataArrayList.of(
                lpoint(ts0 + step * 8, 100),
                lpoint(ts0 + step * 9, 200),
                lpoint(ts0 + step * 10, 300)));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setType(MetricType.RATE);
        expected.addAll(AggrGraphDataArrayList.of(
                lpoint(ts0, 0),
                lpoint(ts0 + step, 700),
                lpoint(ts0 + step * 2, 900),
                lpoint(ts0 + step * 3, 1400),
                lpoint(ts0 + step * 4, 1600),
                lpoint(ts0 + step * 5, 1850),
                lpoint(ts0 + step * 6, 2070),
                lpoint(ts0 + step * 7, 2370),
                lpoint(ts0 + step * 8, 2470),
                lpoint(ts0 + step * 9, 2570),
                lpoint(ts0 + step * 10, 2670)));

        MetricArchiveMutable result = combine(oneGauge, twoGauge, rate);
        assertEqualArchives(expected, result);
        close(oneGauge, twoGauge, rate, expected, result);
    }

    @Test
    public void aggregateSplitOnTwoArchiveWhenKindChanged() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();
        final long ts1 = ts0 + step;
        final long ts2 = ts1 + step;

        MetricArchiveMutable one = new MetricArchiveMutable();
        {
            one.setOwnerShardId(42);
            one.setType(MetricType.DGAUGE);

            // ts0
            one.addRecord(point(ts0, 2.0, true, 1));
            one.addRecord(point(ts0, 3.0, true, 1));
            one.addRecord(point(ts0, 1.0, true, 1));

            // ts1
            one.addRecord(point(ts1, 1.0, true, 1));
            one.addRecord(point(ts1, 2.0, true, 1));
        }

        MetricArchiveMutable two = new MetricArchiveMutable();
        {
            two.setOwnerShardId(42);
            two.setType(MetricType.DGAUGE);

            // ts1, from previous archive
            two.addRecord(point(ts1, 4.0, true, 1));

            // ts2
            two.addRecord(point(ts2, 5.0, true, 1));
            two.addRecord(point(ts2, 2.0, true, 1));
            two.addRecord(point(ts2, 4.0, true, 1));

            // second archive already migrated to new kind
            two.setType(MetricType.RATE);
        }

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setOwnerShardId(42);
        tree.setType(MetricType.RATE);

        MetricArchiveMutable expected = new MetricArchiveMutable();
        {
            expected.setOwnerShardId(42);
            expected.setType(MetricType.RATE);
            expected.addRecord(lpoint(ts0, 60, true, 3));
            expected.addRecord(lpoint(ts1, 130, true, 3));
            expected.addRecord(lpoint(ts2, 240, true, 3));
        }

        MetricArchiveMutable result = combine(one, two);
        assertEqualArchives(expected, result);
        close(one, two, tree, expected, result);
    }

    @Test
    public void overlapsByLastPoint() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();
        final long ts1 = ts0 + step;
        final long ts2 = ts1 + step;
        final long ts3 = ts2 + step;

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.addRecord(point(ts0, 1));
        one.addRecord(point(ts2, 2));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.addRecord(point(ts2, 1));
        two.addRecord(point(ts3, 2));
        two.forceCloseFrame();

        MetricArchiveMutable archive = combine(one, two);
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts0, 1),
            point(ts2, 1),
            point(ts3, 2));

        assertEquals(expected, AggrGraphDataArrayList.of(archive));
        close(one, two, archive);
    }

    @Test
    public void mergeOneOfFrame() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.addRecord(point(ts0 + (step), 1));
        one.addRecord(point(ts0 + (step * 2), 2));
        one.forceCloseFrame();
        one.addRecord(point(ts0 + (step * 3), 3));
        one.addRecord(point(ts0 + (step * 4), 4));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.addRecord(point(ts0 + (step * 4), 1));
        two.addRecord(point(ts0 + (step * 5), 2));
        two.forceCloseFrame();
        two.addRecord(point(ts0 + (step * 6), 3));
        two.addRecord(point(ts0 + (step * 7), 4));
        two.forceCloseFrame();

        MetricArchiveMutable archive = combine(one, two);
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts0 + (step), 1),
            point(ts0 + (step * 2), 2),
            point(ts0 + (step * 3), 3),
            point(ts0 + (step * 4), 1),
            point(ts0 + (step * 5), 2),
            point(ts0 + (step * 6), 3),
            point(ts0 + (step * 7), 4));

        assertEquals(expected, AggrGraphDataArrayList.of(archive));
        close(one, two, archive);
    }

    @Test
    public void combineRandom() {
        var random = ThreadLocalRandom.current();
        var archives = new ArrayList<MetricArchiveMutable>();
        long ts0 = System.currentTimeMillis();
        long last = ts0;
        for (int index = 0; index < random.nextInt(10); index++) {
            var archive = new MetricArchiveMutable();
            long since = random.nextLong(ts0, Math.max(ts0, last) + 1);
            Utils.writeUntilCloseFrame(archive, Math.max(ts0, since));
            last = Math.max(last, archive.getLastTsMillis());
            last += random.nextLong(TimeUnit.SECONDS.toMillis(1), TimeUnit.DAYS.toMillis(30));
            archives.add(archive);
        }

        var result = combine(archives.toArray(new MetricArchiveMutable[0]));
        var expected = MergingAggrGraphDataIterable.of(archives).iterator();
        assertEqualTo(expected, result.iterator());
        close(archives);
        close(result);
    }

    @Test
    public void mergeConcat() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.addRecord(point(ts0 + (step), 1));
        one.addRecord(point(ts0 + (step * 2), 2));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.addRecord(point(ts0 + (step * 3), 3));
        two.addRecord(point(ts0 + (step * 4), 4));

        MetricArchiveMutable archive = combine(one, two);
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts0 + (step), 1),
            point(ts0 + (step * 2), 2),
            point(ts0 + (step * 3), 3),
            point(ts0 + (step * 4), 4));

        assertEquals(expected, AggrGraphDataArrayList.of(archive));
        close(one, two, archive);
    }

    @Test
    public void mergeLastBeforeFirst() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.addRecord(point(ts0 + (step), 1));
        one.addRecord(point(ts0 + (step * 2), 2));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.addRecord(point(ts0 + (step * 3), 3));
        two.addRecord(point(ts0 + (step * 4), 4));

        MetricArchiveMutable archive = combine(two, one);
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts0 + (step), 1),
            point(ts0 + (step * 2), 2),
            point(ts0 + (step * 3), 3),
            point(ts0 + (step * 4), 4));

        assertEquals(expected, AggrGraphDataArrayList.of(archive));
        close(one, two, archive);
    }

    @Test
    public void mergeLastBeforeFirstWithOverlap() {
        final long step = 10_000;
        final long ts0 = Instant.parse("2018-08-20T09:05:00Z").toEpochMilli();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.addRecord(point(ts0 + (step), 1));
        one.addRecord(point(ts0 + (step * 2), 2));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.addRecord(point(ts0 + (step * 2), 1));

        MetricArchiveMutable archive = combine(two, one);
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts0 + (step), 1),
            point(ts0 + (step * 2), 2));

        assertEquals(expected, AggrGraphDataArrayList.of(archive));
        close(one, two, archive);
    }

    @Test
    public void combineDeleteAllAndChangeType() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setOwnerShardId(42);
        one.setOwnerProjectIdEnum(EProjectId.GOLOVAN);
        one.setType(MetricType.LOG_HISTOGRAM);
        one.addRecord(AggrPoint.builder()
                .time("2018-08-20T09:05:30Z")
                .logHistogram(LogHistogram.ofBuckets(1, 2, 3))
                .build());

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setOwnerShardId(42);
        two.setDeleteBefore(DeleteBeforeField.DELETE_ALL);

        MetricArchiveMutable tree = new MetricArchiveMutable();
        tree.setOwnerShardId(42);
        tree.setOwnerProjectIdEnum(EProjectId.GOLOVAN);
        tree.setType(MetricType.HIST);
        tree.addRecord(AggrPoint.builder()
                .time("2018-08-20T09:05:30Z")
                .histogram(dhistogram(new double[]{100, 200, 300}, new long[]{1, 2, 3}))
                .build());

        MetricArchiveMutable result = combine(one, two, tree);
        assertEquals(DeleteBeforeField.DELETE_ALL, result.getDeleteBefore());
        close(one, two, tree, result);
    }

    private static long timeToMillis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private static void assertEqualArchives(MetricArchiveMutable expected, MetricArchiveMutable actual) {
        assertEquals(expected.header(), actual.header());
        assertEquals(AggrGraphDataArrayList.of(expected), AggrGraphDataArrayList.of(actual));
    }

    private MetricArchiveMutable combine(MetricArchiveMutable... archives) {
        List<MetricArchiveImmutable> source = Stream.of(archives)
                .map(MetricArchiveMutable::toImmutableNoCopy)
                .collect(Collectors.toList());

        int expectedBytesEstimation = source.stream()
            .mapToInt(MetricArchiveImmutable::bytesCount)
            .sum();

        long[] lastTssMillis = Stream.of(archives)
            .mapToLong(MetricArchiveMutable::getLastTsMillis)
            .toArray();

        CombineResult combined = ArchiveCombiner.combineArchives(1, 42L, source, lastTssMillis);
        assertEquals(expectedBytesEstimation, combined.getElapsedBytes());
        MetricArchiveMutable result = new MetricArchiveMutable(combined.getHeader());
        result.addAllNoSortMergeFrom(combined.getIt());
        close(source);
        return result;
    }
}
