package ru.yandex.stockpile.server.shard;

import java.util.OptionalLong;

import org.junit.Before;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.memState.LogEntriesContent;
import ru.yandex.stockpile.server.data.log.LogReason;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.tool.BitBufAssume;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class LogStateTest {

    private LogState state;
    private TxTracker txTracker = new TxTracker(42);

    @Before
    public void setUp() {
        state = new LogState(StockpileShardId.random());
    }

    @Test
    public void writeLogTx() {
        assertEquals(OptionalLong.empty(), state.firstTxnSinceLastLogSnapshot());

        final long localId = StockpileLocalId.random();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.addRecord(randomPoint(MetricType.DGAUGE));
        completeWriteLogTx(logEntryContent(localId, one));
        assertArchiveEquals(one, getState(localId));
        assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.addRecord(randomPoint(MetricType.DGAUGE));
        completeWriteLogTx(logEntryContent(localId, two));
        assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());

        MetricArchiveMutable merged = MetricArchiveUtils.merge(one.toImmutable(), two.toImmutable());
        assertArchiveEquals(merged, getState(localId));
        close(one, two, merged);
    }

    @Test(expected = IllegalStateException.class)
    public void takeLogSnapshotEmpty() {
        assertTrue(state.isEmpty());
        state.takeLogSnapshot();
    }

    @Test
    public void logSnapshotNoTxBeforeComplete() {
        var localId = StockpileLocalId.random();
        var mask = randomMask(MetricType.DGAUGE);
        var point = new AggrPoint();

        var expected = new MetricArchiveMutable();

        // txn 43
        {
            var archive = new MetricArchiveMutable();
            archive.addRecord(randomPoint(point, mask));
            expected.addRecord(point);

            completeWriteLogTx(logEntryContent(localId, archive));
            assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());
            assertEquals(expected, getState(localId));
        }

        // txn 44
        {
            var archive = new MetricArchiveMutable();
            archive.addRecord(randomPoint(point, mask));
            expected.addRecord(point);

            completeWriteLogTx(logEntryContent(localId, archive));
            assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());
            assertEquals(expected, getState(localId));
        }

        LogEntriesContent snapshot = state.takeLogSnapshot();
        assertEquals(expected, snapshot.getMetricToArchiveMap().getArchiveRef(localId));
        assertEquals(expected, getState(localId));
        state.completeWriteLogSnapshot(snapshot.sizeAndCount().size(), txTracker.allocateTx());
        assertFalse(state.isEmpty());
        assertEquals(OptionalLong.empty(), state.firstTxnSinceLastLogSnapshot());

        // txn 46
        {
            var archive = new MetricArchiveMutable();
            archive.addRecord(randomPoint(point, mask));

            completeWriteLogTx(logEntryContent(localId, archive));
            assertEquals(OptionalLong.of(46), state.firstTxnSinceLastLogSnapshot());
            assertEquals("snapshot immutable", expected, snapshot.getMetricToArchiveMap().getArchiveRef(localId));

            expected.addRecord(point);
            assertEquals(expected, getState(localId));

            LogEntriesContent snapshotTwo = state.takeLogSnapshot();
            assertEquals("snapshot contains only new txn", archive, snapshotTwo.getMetricToArchiveMap().getArchiveRef(localId));
            snapshotTwo.release();
        }
    }

    @Test
    public void logSnapshotTxBeforeComplete() {
        var localId = StockpileLocalId.random();
        var mask = randomMask(MetricType.DGAUGE);
        var point = RecyclableAggrPoint.newInstance();

        var expected = new MetricArchiveMutable();

        // txn 43
        var txn43 = new MetricArchiveMutable();
        {
            txn43.addRecord(randomPoint(point, mask));
            expected.addRecord(point);

            completeWriteLogTx(logEntryContent(localId, txn43));
            assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());
            assertEquals(expected, getState(localId));
        }

        long snapshotTxn = txTracker.allocateTx();
        LogEntriesContent snapshot = state.takeLogSnapshot();
        assertEquals(txn43, snapshot.getMetricToArchiveMap().getArchiveRef(localId));
        assertEquals(txn43, getState(localId));

        // txn 45
        var txn45 = new MetricArchiveMutable();
        {
            txn45.addRecord(randomPoint(point, mask));
            expected.addRecord(point);

            completeWriteLogTx(logEntryContent(localId, txn45));
            assertEquals(OptionalLong.of(45), state.firstTxnSinceLastLogSnapshot());
            assertEquals(expected, getState(localId));
        }

        state.completeWriteLogSnapshot(snapshot.sizeAndCount().size(), snapshotTxn);
        assertFalse(state.isEmpty());
        assertEquals("snapshot immutable", txn43, snapshot.getMetricToArchiveMap().getArchiveRef(localId));
        assertEquals(OptionalLong.of(45), state.firstTxnSinceLastLogSnapshot());
        assertEquals(expected, getState(localId));

        LogEntriesContent snapshotTwo = state.takeLogSnapshot();
        assertEquals("snapshot contains only new txn", txn45, snapshotTwo.getMetricToArchiveMap().getArchiveRef(localId));
        state.completeWriteLogSnapshot(snapshotTwo.getFileSize(), txTracker.allocateTx());
        assertEquals(expected, getState(localId));
        snapshotTwo.release();
    }

    @Test
    public void anyReasonToLogEmpty() {
        assertEquals(LogReason.TX, state.anyReasonToLog());
        state.requestForceLogSnapshot();
        assertEquals(LogReason.TX, state.anyReasonToLog());
    }

    @Test
    public void twoHoursSnapshot() {
        var localId = StockpileLocalId.random();
        var mask = randomMask(MetricType.DGAUGE);
        var point = new AggrPoint();

        var expected = new MetricArchiveMutable();

        // txn 43
        {
            var archive = new MetricArchiveMutable();
            archive.addRecord(randomPoint(point, mask));
            expected.addRecord(point);

            completeWriteLogTx(logEntryContent(localId, archive));
            assertEquals(OptionalLong.of(43), state.firstTxnSinceLastLogSnapshot());
            assertEquals(expected, getState(localId));
            close(archive);
        }

        assertFalse(state.isEmpty());

        var twoHoursSnapshot = takeTwoHoursSnapshot();
        assertTrue(state.isEmpty());
        assertEquals(expected, twoHoursSnapshot.getMetricToArchiveMap().getArchiveRef(localId));
        assertEquals(new MetricArchiveMutable(), getState(localId));
        assertEquals(OptionalLong.empty(), state.firstTxnSinceLastLogSnapshot());
        twoHoursSnapshot.release();
        close(expected);
    }

    @Test
    public void writeLogTxIgnoreInvalidMetric() {
        final long localId = StockpileLocalId.random();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.addRecord(randomPoint(MetricType.DGAUGE));
        completeWriteLogTx(logEntryContent(localId, one));
        assertArchiveEquals(one, getState(localId));

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.HIST);
        two.addRecord(randomPoint(MetricType.HIST));
        completeWriteLogTx(logEntryContent(localId, two));

        // invalid write event if it was successful wrote should be send to /dev/null
        assertArchiveEquals(one, getState(localId));
        close(one, two);
    }

    @Test
    public void objectSizeEmpty() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        LogState logState = new LogState(1);

        GraphLayout gl = GraphLayout.parseInstance(logState);
        assertEquals(gl.toPrintable(), gl.totalSize(), logState.memorySizeIncludingSelf(), 500);
    }

    @Test
    public void objectSize() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        LogState logState = new LogState(42);

        for (int index = 1; index < 100; index++) {
            StockpileLogEntryContent s = new StockpileLogEntryContent();
            s.addAggrPoint(index % 10, randomPoint(MetricType.DGAUGE));
            logState.completeWriteLogTx(s, s.estimateSerializedSize(), index + 100);
            s.release();

            if (index == 60) {
                logState.takeLogSnapshot().release();
            }
        }

        // Exclude enums because exists into single instance into app
        long enumSize = GraphLayout.parseInstance(MetricType.DGAUGE, StockpileFormat.CURRENT).totalSize();

        GraphLayout gl = GraphLayout.parseInstance(logState);
        assertEquals(gl.totalSize() - enumSize, logState.memorySizeIncludingSelf(), 1000);
    }

    private MetricArchiveMutable getState(long localId) {
        MetricArchiveMutable result = new MetricArchiveMutable();
        state.contentSinceLastSnapshotStream(localId).forEach(result::updateWith);
        return result;
    }

    private LogEntriesContent takeTwoHoursSnapshot() {
        return state.takeForTwoHourSnapshot();
    }

    private void completeWriteLogTx(StockpileLogEntryContent log) {
        state.completeWriteLogTx(log, log.estimateSerializedSize(), txTracker.allocateTx());
    }

    private static StockpileLogEntryContent logEntryContent(long localId, MetricArchiveMutable archive) {
        StockpileLogEntryContent content = new StockpileLogEntryContent();
        content.addMetricMultiArchive(localId, archive);
        return content;
    }

    private static void assertArchiveEquals(MetricArchiveMutable expected, MetricArchiveMutable actual) {
        assertEquals(expected.header(), actual.header());
        assertEquals(expected.toAggrGraphDataArrayList(), actual.toAggrGraphDataArrayList());
    }
}
