package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.SubmissionPublisher;
import java.util.concurrent.TimeUnit;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.data.DeletedShardSet;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeDeletedShardFilterTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    @Test
    public void safeAll() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(2, 42);

        var source = new SubmissionPublisher<MetricIdAndData>();
        var filter = new MergeDeletedShardFilter(deletedShards);
        var target = new Consumer();

        source.subscribe(filter);
        filter.subscribe(target);

        var expected = List.of(randomData(1, 1));
        expected.forEach(source::submit);
        source.close();

        assertEquals(expected, target.future.join());
    }

    @Test
    public void safeUnknownShard() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 42);

        var source = new SubmissionPublisher<MetricIdAndData>();
        var filter = new MergeDeletedShardFilter(deletedShards);
        var target = new Consumer();

        source.subscribe(filter);
        filter.subscribe(target);

        var expected = List.of(randomData(0, 0));
        expected.forEach(source::submit);
        source.close();

        assertEquals(expected, target.future.join());
    }

    @Test
    public void deleteAll() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 2);
        deletedShards.add(2, 3);

        var source = new SubmissionPublisher<MetricIdAndData>();
        var filter = new MergeDeletedShardFilter(deletedShards);
        var target = new Consumer();

        source.subscribe(filter);
        filter.subscribe(target);

        var all = List.of(
                randomData(1, 2),
                randomData(2, 3),
                randomData(1, 2),
                randomData(2, 3));
        all.forEach(source::submit);
        source.close();

        assertEquals(new ArrayList<>(), target.future.join());
    }

    @Test
    public void filter() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 42);
        deletedShards.add(2, 43);

        var source = new SubmissionPublisher<MetricIdAndData>();
        var filter = new MergeDeletedShardFilter(deletedShards);
        var target = new Consumer();

        source.subscribe(filter);
        filter.subscribe(target);

        List<MetricIdAndData> all = new ArrayList<>();
        List<MetricIdAndData> expected = new ArrayList<>();

        {
            var data = randomData(1, 41);
            all.add(data);
            expected.add(data);
        }
        {
            var data = randomData(1, 42);
            all.add(data);
        }
        {
            var data = randomData(1, 43);
            all.add(data);
            expected.add(data);
        }
        {
            var data = randomData(2, 42);
            all.add(data);
            expected.add(data);
        }
        {
            var data = randomData(2, 43);
            all.add(data);
        }

        all.forEach(source::submit);
        source.close();

        assertEquals(expected, target.future.join());
    }

    private MetricIdAndData randomData(int projectId, int shardId) {
        var archive = new MetricArchiveMutable();
        archive.addRecord(randomPoint());
        archive.setOwnerProjectId(projectId);
        archive.setOwnerShardId(shardId);

        return new MetricIdAndData(
                StockpileLocalId.random(),
                archive.getLastTsMillis(),
                0,
                archive.toImmutableNoCopy()
        );
    }

    private static class Consumer implements Flow.Subscriber<MetricIdAndData> {
        private final List<MetricIdAndData> items = new ArrayList<>();
        private final CompletableFuture<List<MetricIdAndData>> future = new CompletableFuture<>();
        private Flow.Subscription subscription;

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.subscription = subscription;
            subscription.request(5);
        }

        @Override
        public void onNext(MetricIdAndData item) {
            subscription.request(1);
            items.add(item);
        }

        @Override
        public void onError(Throwable e) {
            future.completeExceptionally(e);
        }

        @Override
        public void onComplete() {
            future.complete(items);
        }
    }
}
