package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Flow;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;

import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.CloseableUtils;
import ru.yandex.solomon.util.time.TimeProvider;
import ru.yandex.solomon.util.time.TimeProviderTestImpl;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricIdAndData;

import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeMergerTest {
    private static final Logger logger = LoggerFactory.getLogger(MergeMergerTest.class);

    private ExecutorService executor = ForkJoinPool.commonPool();
    private TimeProvider clock = new TimeProviderTestImpl();

    @Test
    public void empty() {
        var result = run(List.of(), MergeKind.DAILY);
        assertEquals(List.of(), result);
    }

    @Test
    public void one() {
        Metric metric = ransomMetric(10);
        List<MergeTaskResult> result = run(List.of(metric), MergeKind.DAILY);

        assertEquals(1, result.size());
        assertEquals(metric.localId, result.get(0).getCurrentLevel().localId());
        assertEquals(metric.merged(), result.get(0).getCurrentLevel().archive());
        close(metric);
    }

    @Test
    public void orderSaved() {
        Metric source = ransomMetric(20);
        List<Metric> metrics = new ArrayList<>(1000);
        for (int index = 0; index < 1000; index++) {
            metrics.add(source.duplicate(StockpileLocalId.random()));
        }

        long[] expected = metrics.stream().mapToLong(value -> value.localId).toArray();
        long[] result = run(metrics, MergeKind.DAILY)
            .stream()
            .mapToLong(mergeTaskResult -> mergeTaskResult.getCurrentLevel().localId())
            .toArray();

        assertArrayEquals(expected, result);
        close(metrics);
    }

    @Test
    public void dontOverflowItSelf() throws InterruptedException {
        Metric source = ransomMetric(1);
        var producer = new InfinityPublisher(source);
        var merger = new MergeMerger(42, executor, clock.nowMillis(), false, 0,
                InvalidArchiveStrategy.fail(),
                new MergeProcessMetrics().getMergeKindMetrics(MergeKind.DAILY));
        var consumer = new SlowConsumer<MergeTaskResult>(1);

        CountDownLatch sync = consumer.sync.get();
        merger.subscribe(consumer);
        producer.subscribe(merger);

        sync.await();
        assertEquals(1, consumer.consumed.get());

        for (int index = 2; index < 10; index++) {
            consumer.awaitNext();
            assertEquals(index, consumer.consumed.get());
        }
    }

    @Test
    public void memoryUsage() throws InterruptedException {
        var producer = new InfinityPublisher(ransomMetric(ThreadLocalRandom.current().nextInt(0, 2)));
        var merger = new MergeMerger(42, executor, clock.nowMillis(), false, 0,
                InvalidArchiveStrategy.fail(),
                new MergeProcessMetrics().getMergeKindMetrics(MergeKind.DAILY));
        var alice = new SlowConsumer<MergeTaskResult>(0);
        var bob = new SlowConsumer<MergeTaskResult>(0);

        var initMemory = merger.memorySizeIncludingSelf();
        merger.subscribe(alice);
        merger.subscribe(bob);

        producer.subscribe(merger);

        alice.awaitNext();
        assertThat(merger.memorySizeIncludingSelf(), greaterThan(initMemory));

        for (int index = 0; index < 100; index++) {
            alice.awaitNext();
            bob.awaitNext();
        }

        assertThat(merger.memorySizeIncludingSelf(), greaterThan(initMemory));
    }

    private List<MergeTaskResult> run(List<Metric> metrics, MergeKind kind) {
        boolean allowDecim = kind == MergeKind.ETERNITY;
        boolean allowDelete = kind == MergeKind.ETERNITY;
        var producer = new Publisher(metrics);
        var merger = new MergeMerger(42, executor, clock.nowMillis(), allowDecim, 0,
                InvalidArchiveStrategy.fail(),
                new MergeProcessMetrics().getMergeKindMetrics(kind));
        var initMemory = merger.memorySizeIncludingSelf();
        var consumer = new Consumer();

        merger.subscribe(consumer);
        producer.subscribe(merger);
        var result = consumer.future.join();
        assertEquals(initMemory, merger.memorySizeIncludingSelf());
        return result;
    }


    private Metric ransomMetric(int countPoints) {
        long localId = StockpileLocalId.random();
        AggrPoint point = new AggrPoint();
        List<MetricArchiveMutable> archives = new ArrayList<>(3);
        long now = System.currentTimeMillis();
        for (int index = 0; index < 3; index++) {
            MetricArchiveMutable archive = new MetricArchiveMutable();
            archive.setType(MetricType.DGAUGE);

            for (int pointIndex = 0; pointIndex < countPoints; pointIndex++) {
                now += 10_000;
                point.setTsMillis(now);
                point.setValue(ValueRandomData.randomNum(ThreadLocalRandom.current()));
                archive.addRecord(point);
            }

            archives.add(archive);
        }

        return new Metric(localId, archives);
    }

    private static class Metric implements AutoCloseable {
        private final long localId;
        private final List<MetricArchiveMutable> archives;

        public Metric(long localId, List<MetricArchiveMutable> archives) {
            this.localId = localId;
            this.archives = archives;
        }

        private List<MetricIdAndData> prepare() {
            var result = new ArrayList<MetricIdAndData>(archives.size());
            for (int index = 0; index < archives.size(); index++) {
                var archive = archives.get(index);
                result.add(new MetricIdAndData(localId, archive.getLastTsMillis(), 0, archive.toImmutableNoCopy()));
            }
            return result;
        }

        public MetricArchiveImmutable merged() {
            try (MetricArchiveMutable result = new MetricArchiveMutable()) {
                for (var archive : archives) {
                    result.updateWith(archive);
                }
                result.sortAndMerge();
                return result.toImmutableNoCopy();
            }
        }

        public Metric duplicate(long localId) {
            return new Metric(localId, archives);
        }

        @Override
        public void close() {
            CloseableUtils.close(archives);
        }
    }

    private static class Consumer implements Flow.Subscriber<MergeTaskResult> {
        private List<MergeTaskResult> items = new ArrayList<>();
        private Flow.Subscription subscription;
        private CompletableFuture<List<MergeTaskResult>> future = new CompletableFuture<>();

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.subscription = subscription;
            subscription.request(5);
        }

        @Override
        public void onNext(MergeTaskResult item) {
            subscription.request(1);
            items.add(item);
        }

        @Override
        public void onError(Throwable e) {
            future.completeExceptionally(e);
        }

        @Override
        public void onComplete() {
            future.complete(items);
        }
    }

    private class Publisher implements Flow.Publisher<List<MetricIdAndData>> {
        private AtomicLong requested = new AtomicLong();
        private Flow.Subscriber<? super List<MetricIdAndData>> subscriber;
        private ActorRunner actor;
        private int index = 0;
        private List<Metric> metrics;
        private boolean doneCalled = false;

        public Publisher(List<Metric> metrics) {
            this.metrics = metrics;
            actor = new ActorRunner(this::act, executor, throwable -> subscriber.onError(throwable));
        }

        @Override
        public void subscribe(Flow.Subscriber<? super List<MetricIdAndData>> subscriber) {
            this.subscriber = subscriber;
            subscriber.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    requested.addAndGet(n);
                    actor.schedule();
                }

                @Override
                public void cancel() {
                    requested.set(0);
                    actor.schedule();
                }
            });
        }

        private void act() {
            if (requested.get() <= 0 || doneCalled) {
                return;
            }

            while (requested.get() > 0) {
                if (index == metrics.size()) {
                    doneCalled = true;
                    subscriber.onComplete();
                    return;
                }

                requested.decrementAndGet();
                var metric = metrics.get(index++);
                subscriber.onNext(metric.prepare());
            }
        }
    }

    private class InfinityPublisher implements Flow.Publisher<List<MetricIdAndData>> {
        private AtomicLong produced = new AtomicLong();
        private AtomicLong requested = new AtomicLong();
        private Flow.Subscriber<? super List<MetricIdAndData>> subscriber;
        private ActorRunner actor;
        private Metric metric;
        private volatile boolean done = false;
        private volatile boolean doneCalled = false;
        private volatile CountDownLatch sync = new CountDownLatch(1);

        public InfinityPublisher(Metric metric) {
            this.metric = metric;
            this.actor = new ActorRunner(this::act, executor, throwable -> subscriber.onError(throwable));
        }

        @Override
        public void subscribe(Flow.Subscriber<? super List<MetricIdAndData>> subscriber) {
            this.subscriber = subscriber;
            subscriber.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    logger.debug("requested from producer: {}, total produced: {}", n, produced.get());
                    requested.addAndGet(n);
                    actor.schedule();
                }

                @Override
                public void cancel() {
                    requested.set(0);
                    actor.schedule();
                }
            });
        }

        private void act() {
            try {
                if (requested.get() <= 0 || doneCalled) {
                    return;
                }

                if (done) {
                    subscriber.onComplete();
                    doneCalled = true;
                    return;
                }

                while (requested.get() > 0) {
                    produced.incrementAndGet();
                    subscriber.onNext(metric.prepare());
                    requested.decrementAndGet();
                }
            } finally {
                var copy = sync;
                sync = new CountDownLatch(1);
                copy.countDown();
            }
        }

        public void awaitAct() throws InterruptedException {
            var copy = sync;
            actor.schedule();
            copy.await();
        }

        private void complete() {
            done = true;
        }
    }
}
