package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringJoiner;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.CloseableUtils;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.actor.ActorRunnableType;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestBase;

import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeReaderTest extends StockpileShardTestBase {
    @Before
    public void setUp() {
        restart();
    }

    @Test
    public void one() {
        Metric metric = ransomMetric(2);
        pushAndSnapshotSync(List.of(metric));

        List<Metric> result = read();
        assertEquals(1, result.size());
        assertEquals(metric, result.get(0));
        close(metric);
        close(result);
    }

    @Test
    public void readFew() {
        List<Metric> metrics = IntStream.range(0, 3)
            .mapToObj(value -> ransomMetric(2))
            .peek(metric -> pushAndSnapshotSync(ImmutableList.of(metric)))
            .collect(Collectors.toList());

        metrics.sort((o1, o2) -> Long.compareUnsigned(o1.localId, o2.localId));

        List<Metric> result = read();
        assertEquals(metrics, result);
        close(metrics);
        close(result);
    }

    @Test
    public void readLocalIdPositiveOrdered() {
        List<Metric> metrics = IntStream.range(1, ThreadLocalRandom.current().nextInt(100, 10000))
            .mapToObj(localId -> ransomMetric(10).duplicate(localId))
            .collect(Collectors.toList());

        Collections.shuffle(metrics);

        int snapshots = ThreadLocalRandom.current().nextInt(2, 50);

        Lists.partition(metrics, snapshots)
            .forEach(this::pushAndSnapshotSync);

        metrics.sort((o1, o2) -> StockpileLocalId.compare(o1.localId, o2.localId));

        var result = read().toArray(new Metric[0]);
        assertArrayEquals(metrics.toArray(), result);
        close(metrics);
        close(result);
    }

    @Test
    public void readLocalIdNegativeOrdered() {
        List<Metric> metrics = IntStream.range(1, ThreadLocalRandom.current().nextInt(100, 10000))
            .mapToObj(localId -> ransomMetric(10).duplicate(Long.MAX_VALUE + localId))
            .collect(Collectors.toList());

        Collections.shuffle(metrics);

        int snapshots = ThreadLocalRandom.current().nextInt(2, 50);

        Lists.partition(metrics, snapshots)
            .forEach(this::pushAndSnapshotSync);

        metrics.sort((o1, o2) -> StockpileLocalId.compare(o1.localId, o2.localId));

        var result = read().toArray(new Metric[0]);
        assertArrayEquals(metrics.toArray(), result);
        close(metrics);
        close(result);
    }

    @Test
    public void localIdsSorted() {
        Metric proto = ransomMetric(5);
        List<Metric> metrics = IntStream.range(0, 1000)
            .mapToObj(value -> proto.duplicate(StockpileLocalId.random()))
            .collect(Collectors.toList());
        Collections.shuffle(metrics);

        for (var part : Lists.partition(metrics, 250)) {
            pushAndSnapshotSync(part);
        }

        metrics.sort((o1, o2) -> Long.compareUnsigned(o1.localId, o2.localId));
        long[] expected = metrics.stream().mapToLong(value -> value.localId).toArray();
        var read = read();
        long[] result = read.stream().mapToLong(value -> value.localId).toArray();
        assertArrayEquals(expected, result);
        close(metrics);
        close(read);
    }

    @Test
    public void raceOnRead() throws InterruptedException {
        Metric proto = ransomMetric(5);
        List<Metric> metrics = IntStream.range(0, 10_000)
            .mapToObj(value -> proto.duplicate(StockpileLocalId.random()))
            .collect(Collectors.toList());
        Collections.shuffle(metrics);

        for (var part : Lists.partition(metrics, 5_000)) {
            pushAndSnapshotSync(part);
        }
        close(proto);

        var reader = createMergeReader();
        var consumer = new SlowConsumer<>(100);
        reader.subscribe(consumer);

        while (true) {
            var sync = consumer.sync.get();
            if (consumer.consumed.get() >= 100) {
                break;
            }
            sync.await();
        }

        assertEquals(100, consumer.consumed.get());

        for (int index = 100; index < 10_000; index++) {
            var sync = consumer.sync.get();
            // trigger loading, but nothing
            consumer.subscription.request(1);
            assertTrue(sync.await(1, TimeUnit.SECONDS));
            assertEquals(index + 1, consumer.consumed.get());
        }
    }

    @Test
    public void memoryUsage() throws InterruptedException {
        List<Metric> metrics = IntStream.range(0, 10000)
            .mapToObj(value -> ransomMetric(ThreadLocalRandom.current().nextInt(1)))
            .collect(Collectors.toList());
        pushAndSnapshotSync(metrics);
        close(metrics);

        MergeReader reader = createMergeReader();
        long initMemory = reader.memorySizeIncludingSelf();
        assertThat(initMemory, greaterThan(0L));

        var consumer = new SlowConsumer<List<MetricIdAndData>>(0);
        reader.subscribe(consumer);

        consumer.awaitNext();
        assertThat(reader.memorySizeIncludingSelf(), greaterThan(initMemory));

        for (int index = 0; index < 100; index++) {
            consumer.awaitNext();
        }
        assertThat(reader.memorySizeIncludingSelf(), greaterThan(initMemory));
    }

    private SnapshotIndex[] getIndexes() {
        CompletableFuture<SnapshotIndex[]> future = new CompletableFuture<>();
        stockpileShard.run(ActorRunnableType.MISC, actor ->
            future.complete(stockpileShard.stateDone()
                .indexes(actor)
                .toArray(SnapshotIndex[]::new)));
        return future.join();
    }

    public List<Metric> read() {
        MergeReader reader = createMergeReader();
        long initMemory = reader.memorySizeIncludingSelf();
        assertThat(initMemory, greaterThan(0L));

        Consumer consumer = new Consumer();
        reader.subscribe(consumer);
        var result = consumer.future.join();
        assertEquals(initMemory, reader.memorySizeIncludingSelf());
        return result;
    }

    private MergeReader createMergeReader() {
        return new MergeReader(
            new ShardThreadStub(stockpileShard),
            getIndexes(),
            new MergeProcessMetrics().getMergeKindMetrics(MergeKind.DAILY));
    }

    public void pushAndSnapshotSync(List<Metric> metrics) {
        for (int index = 0; index < metrics.get(0).archives.size(); index++) {
            var builder = StockpileWriteRequest.newBuilder();
            for (Metric metric : metrics) {
                builder.addArchiveCopy(metric.localId, metric.archives.get(index));
            }

            stockpileShard.pushBatch(builder.build()).join();
            stockpileShard.forceSnapshot().join();
        }
    }

    private Metric ransomMetric(int countPoints) {
        long localId = StockpileLocalId.random();
        AggrPoint point = new AggrPoint();
        List<MetricArchiveMutable> archives = new ArrayList<>(3);
        long now = System.currentTimeMillis();
        for (int index = 0; index < 3; index++) {
            MetricArchiveMutable archive = new MetricArchiveMutable();
            archive.setType(MetricType.DGAUGE);

            for (int pointIndex = 0; pointIndex < countPoints; pointIndex++) {
                now += 10_000;
                point.setTsMillis(now);
                point.setValue(ValueRandomData.randomNum(ThreadLocalRandom.current()));
                archive.addRecord(point);
            }

            archives.add(archive);
        }

        return new Metric(localId, archives);
    }

    private static class Metric implements AutoCloseable {
        private final long localId;
        private final List<MetricArchiveMutable> archives;

        public Metric(long localId, List<MetricArchiveMutable> archives) {
            this.localId = localId;
            this.archives = archives;
        }

        public Metric duplicate(long localId) {
            return new Metric(localId, archives);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Metric metric = (Metric) o;

            if (localId != metric.localId) return false;
            return archives.equals(metric.archives);
        }

        @Override
        public int hashCode() {
            int result = (int) (localId ^ (localId >>> 32));
            result = 31 * result + archives.hashCode();
            return result;
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", Metric.class.getSimpleName() + "[", "]")
                .add("localId=" + localId)
                .add("archives=" + archives)
                .toString();
        }

        @Override
        public void close() {
            CloseableUtils.close(archives);
        }
    }

    private static class Consumer implements Flow.Subscriber<List<MetricIdAndData>> {
        private List<List<MetricIdAndData>> items = new ArrayList<>();
        private Flow.Subscription subscription;
        private CompletableFuture<List<Metric>> future = new CompletableFuture<>();

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.subscription = subscription;
            subscription.request(5);
        }

        @Override
        public void onNext(List<MetricIdAndData> item) {
            subscription.request(1);
            items.add(item);
        }

        @Override
        public void onError(Throwable e) {
            future.completeExceptionally(e);
        }

        @Override
        public void onComplete() {
            future.complete(items.stream()
                .map(item -> {
                    var archives = item.stream()
                        .map(data -> data.archive().cloneToUnsealed())
                        .collect(Collectors.toList());

                    return new Metric(item.get(0).localId(), archives);
                })
                .collect(Collectors.toList()));
        }
    }
}
