package ru.yandex.stockpile.server.shard;

import java.time.Clock;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.dataSize.DataSizeUnit;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.ChunkIndexArray;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexProperties;
import ru.yandex.stockpile.server.shard.stat.LevelSizeAndCount;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.stockpile.server.SnapshotLevel.DAILY;
import static ru.yandex.stockpile.server.SnapshotLevel.ETERNITY;
import static ru.yandex.stockpile.server.SnapshotLevel.TWO_HOURS;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeStrategyTest {
    private long txn = 1;

    @Test
    public void oldDaily2h() {
        var opts = MergeOptions.newBuilder()
            .enableNew(false)
            .build();

        var eternity = index(ETERNITY, 32 << 20);
        var daily = index(DAILY, 16 << 20);
        var twoHours1 = index(TWO_HOURS, 1 << 20);
        var twoHours2 = index(TWO_HOURS, 1 << 20);

        var result = merge(opts, MergeKind.DAILY, eternity, daily, twoHours1, twoHours2);
        assertIndexEquals(result.indexes, daily, twoHours1, twoHours2);
        assertFalse(result.allowDecim);
        assertFalse(result.allowDelete);
        assertEquals(0, result.splitDelayMillis);
    }

    @Test
    public void oldEternityDaily2h() {
        var opts = MergeOptions.newBuilder()
            .enableNew(false)
            .build();

        var eternity = index(ETERNITY, 32 << 20);
        var daily = index(DAILY, 16 << 20);
        var twoHours1 = index(TWO_HOURS, 1 << 20);
        var twoHours2 = index(TWO_HOURS, 1 << 20);

        var result = merge(opts, MergeKind.ETERNITY, eternity, daily, twoHours1, twoHours2);
        assertIndexEquals(result.indexes, eternity, daily, twoHours1, twoHours2);
        assertTrue(result.allowDelete);
        assertTrue(result.allowDecim);
        assertEquals(0, result.splitDelayMillis);
    }

    @Test
    public void firstAtDaily() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .build();

        var eternity = index(ETERNITY, 32 << 20);
        var twoHours1 = index(TWO_HOURS, 1 << 20);
        var twoHours2 = index(TWO_HOURS, 1 << 20);

        var result = merge(opts, MergeKind.DAILY, eternity, twoHours1, twoHours2);
        assertIndexEquals(result.indexes, twoHours1, twoHours2);
        assertFalse(result.allowDelete);
        assertEquals(0, result.splitDelayMillis);
    }

    @Test
    public void firstAtEternity() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .build();

        var daily = index(DAILY, 16 << 20);
        var twoHours1 = index(TWO_HOURS, 1 << 20);
        var twoHours2 = index(TWO_HOURS, 1 << 20);
        var result = merge(opts, MergeKind.ETERNITY, daily, twoHours1, twoHours2);
        assertIndexEquals(result.indexes, daily, twoHours1, twoHours2);
        assertTrue(result.allowDelete);
        assertFalse(result.allowDecim);
        assertEquals(0, result.splitDelayMillis);
    }

    @Test
    public void limit1AtLevel() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(1)
            .allowDecim(true)
            .build();

        var eternity = index(ETERNITY, 32 << 20);
        var daily = index(DAILY, 16 << 20);
        var twoHours1 = index(TWO_HOURS, 1 << 20);
        var twoHours2 = index(TWO_HOURS, 1 << 20);

        var resultDaily = merge(opts, MergeKind.DAILY, eternity, daily, twoHours1, twoHours2);
        assertIndexEquals(resultDaily.indexes, daily, twoHours1, twoHours2);
        assertEquals(0, resultDaily.splitDelayMillis);

        var resultEternity = merge(opts, MergeKind.ETERNITY, eternity, daily, twoHours1, twoHours2);
        assertIndexEquals(resultEternity.indexes, eternity, daily, twoHours1, twoHours2);
        assertTrue(resultEternity.allowDelete);
        assertTrue(resultEternity.allowDecim);
        assertEquals(0, resultEternity.splitDelayMillis);
    }

    @Test
    public void limit2AtLevel() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(2)
            .minSnapshotBytesSize(0) // disable
            .build();

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily = index(DAILY, 16 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily = index(DAILY, 16 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity, daily, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily1 = index(DAILY, 32 << 20);
            var daily2 = index(DAILY, 8 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY, 32 << 20);
            var eternity2 = index(ETERNITY, 2 << 20);
            var daily1 = index(DAILY, 4 << 20);
            var daily2 = index(DAILY, 1 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, eternity2, daily1, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void disableSnapshotLimit() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(0)
            .minSnapshotBytesSize(0)
            .build();

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily1 = index(DAILY, 32 << 20);
            var daily2 = index(DAILY, 8 << 20);
            var daily3 = index(DAILY, 4 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY, 1000 << 20);
            var eternity2 = index(ETERNITY, 200 << 20);
            var eternity3 = index(ETERNITY, 100 << 20);
            var eternity4 = index(ETERNITY, 10 << 20);
            var eternity5 = index(ETERNITY, 5 << 20);
            var daily = index(DAILY, 1 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, eternity3, eternity4, eternity5, daily, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void minSnapshotBytes() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(0) // disable limit snapshots
            .minSnapshotBytesSize(10 << 20) // snapshots less then 10 MiB will be always merged
            .build();

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily1 = index(DAILY, 16 << 20);
            var daily2 = index(DAILY, 4 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY, 32 << 20);
            var eternity2 = index(ETERNITY, 8 << 20);
            var daily = index(DAILY, 4 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, daily, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, eternity2, daily, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void mergeWhenRightMoreThenLeft() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(0) // disable
            .minSnapshotBytesSize(0) // disable
            .build();

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily = index(DAILY,1 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily1 = index(DAILY,10 << 20);
            var daily2 = index(DAILY,1 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 2 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, 32 << 20);
            var daily1 = index(DAILY,32 << 20);
            var daily2 = index(DAILY,4 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, 1 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, eternity, twoHours1, twoHours2);
            assertTrue(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY,8 << 20);
            var eternity2 = index(ETERNITY,2 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 2 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, eternity2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY,32 << 20);
            var eternity2 = index(ETERNITY,16 << 20);
            var twoHours1 = index(TWO_HOURS, 1 << 20);
            var twoHours2 = index(TWO_HOURS, 1 << 20);

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void mergeNoChunks() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(0) // disable
            .minSnapshotBytesSize(0) // disable
            .build();

        {
            var eternity = index(ETERNITY,8 << 20);
            var result = merge(opts, MergeKind.ETERNITY, eternity);
            assertIndexEquals(result.indexes);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity1 = index(ETERNITY, 8 << 20);
            var eternity2 = index(ETERNITY, 4 << 20);
            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2);
            assertIndexEquals(result.indexes);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void weightRightOverflow() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(1, DataSizeUnit.GIGABYTES)
            .build();

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(15).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(8).toBytes());
            var daily3 = index(DAILY, DataSize.fromGigaBytes(3).toBytes());
            var daily4 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var daily5 = index(DAILY, DataSize.fromGigaBytes(13).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());
            var twoHours3 = index(TWO_HOURS, DataSize.fromMegaBytes(318).toBytes());
            var twoHours4 = index(TWO_HOURS, DataSize.fromMegaBytes(581).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, daily4, daily5, twoHours1, twoHours2, twoHours3, twoHours4);
            assertIndexEquals(result.indexes, daily1, daily2, daily3, daily4, daily5, twoHours1, twoHours2, twoHours3, twoHours4);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void splitDelayAllowOnlyForFirstSnapshotAtLevel() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(2, DataSizeUnit.GIGABYTES)
            .splitDelayMillis(1, TimeUnit.DAYS)
            .build();

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(5).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily1, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(TimeUnit.DAYS.toMillis(1), result.splitDelayMillis);
        }
    }

    @Test
    public void forceAfterMillis() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(1, DataSizeUnit.GIGABYTES)
            .forceMergeAfterMillis(5, TimeUnit.DAYS)
            .build();

        {
            // all snapshots young
            long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(4);

            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes(), ts0);
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes(), ts0 + TimeUnit.DAYS.toMillis(1));
            var daily3 = index(DAILY, DataSize.fromGigaBytes(3).toBytes(), ts0 + TimeUnit.DAYS.toMillis(2));
            var daily4 = index(DAILY, DataSize.fromGigaBytes(1).toBytes(), ts0 + TimeUnit.DAYS.toMillis(3));
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, daily4, twoHours1);
            assertIndexEquals(result.indexes, twoHours1);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            // too old snapshot
            long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(6);

            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes(), ts0);
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes(), ts0 + TimeUnit.DAYS.toMillis(1));
            var daily3 = index(DAILY, DataSize.fromGigaBytes(3).toBytes(), ts0 + TimeUnit.DAYS.toMillis(2));
            var daily4 = index(DAILY, DataSize.fromGigaBytes(1).toBytes(), ts0 + TimeUnit.DAYS.toMillis(3));
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, daily4, twoHours1);
            assertIndexEquals(result.indexes, daily1, daily2, daily3, daily4, twoHours1);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            // too old snapshot
            long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(10);
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes(), ts0);
            var result = merge(opts, MergeKind.ETERNITY, eternity);
            assertIndexEquals(result.indexes, eternity);
            assertTrue(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void forceAfterMillisJitter() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(1, DataSizeUnit.GIGABYTES)
            .forceMergeAfterMillis(5, TimeUnit.DAYS)
            .forceMergeAfterJitterMillis(1, TimeUnit.DAYS)
            .build();

        {
            // all snapshots young
            long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(4);

            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes(), ts0);
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes(), ts0 + TimeUnit.DAYS.toMillis(1));
            var daily3 = index(DAILY, DataSize.fromGigaBytes(3).toBytes(), ts0 + TimeUnit.DAYS.toMillis(2));
            var daily4 = index(DAILY, DataSize.fromGigaBytes(1).toBytes(), ts0 + TimeUnit.DAYS.toMillis(3));
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, daily4, twoHours1);
            assertIndexEquals(result.indexes, twoHours1);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            // too old snapshot
            long ts0 = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(5) - TimeUnit.HOURS.toMillis(12);

            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes(), ts0);
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes(), ts0 + TimeUnit.DAYS.toMillis(1));
            var daily3 = index(DAILY, DataSize.fromGigaBytes(3).toBytes(), ts0 + TimeUnit.DAYS.toMillis(2));
            var daily4 = index(DAILY, DataSize.fromGigaBytes(1).toBytes(), ts0 + TimeUnit.DAYS.toMillis(3));
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());

            Map<Integer, Long> counts = IntStream.range(0, 1000)
                .mapToObj(ignore -> {
                    var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, daily4, twoHours1);
                    return result.indexes.length;
                })
                .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
            assertNotEquals(Long.valueOf(0), counts.get(5));
            assertNotEquals(Long.valueOf(0), counts.get(1));
        }
    }

    @Test
    public void avoidSelfPatchMerge() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(1, DataSizeUnit.GIGABYTES)
            .build();

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes());
            var daily3 = index(DAILY, DataSize.fromMegaBytes(150).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3);
            assertIndexEquals(result.indexes);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(40).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(6).toBytes());
            var daily3 = index(DAILY, DataSize.fromMegaBytes(150).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(50).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, daily3, twoHours1);
            assertIndexEquals(result.indexes, daily3, twoHours1);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }
    }

    @Test
    public void allowDecim() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(2, DataSizeUnit.GIGABYTES)
            .splitDelayMillis(1, TimeUnit.DAYS)
            .allowDecim(true)
            .build();

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertFalse(result.allowDecim);
            assertEquals(0, result.splitDelayMillis);
        }

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(5).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.DAILY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, daily1, daily2, twoHours1, twoHours2);
            assertFalse(result.allowDelete);
            assertTrue(result.allowDecim);
            assertEquals(TimeUnit.DAYS.toMillis(1), result.splitDelayMillis);
        }
    }

    @Test
    public void usePrevLabels() {
        var opts = MergeOptions.newBuilder()
            .enableNew(true)
            .snapshotsLimit(5)
            .minSnapshotBytesSize(2, DataSizeUnit.GIGABYTES)
            .usePrevLevels(false)
            .build();

        {
            var eternity = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.ETERNITY, eternity, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes);
        }

        {
            var eternity1 = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var eternity2 = index(ETERNITY, DataSize.fromGigaBytes(10).toBytes());
            var eternity3 = index(ETERNITY, DataSize.fromGigaBytes(5).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, eternity3, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes);
        }

        {
            var eternity1 = index(ETERNITY, DataSize.fromGigaBytes(72).toBytes());
            var eternity2 = index(ETERNITY, DataSize.fromGigaBytes(70).toBytes());
            var eternity3 = index(ETERNITY, DataSize.fromGigaBytes(5).toBytes());
            var daily1 = index(DAILY, DataSize.fromGigaBytes(4).toBytes());
            var daily2 = index(DAILY, DataSize.fromGigaBytes(1).toBytes());
            var twoHours1 = index(TWO_HOURS, DataSize.fromMegaBytes(688).toBytes());
            var twoHours2 = index(TWO_HOURS, DataSize.fromMegaBytes(685).toBytes());

            var result = merge(opts, MergeKind.ETERNITY, eternity1, eternity2, eternity3, daily1, daily2, twoHours1, twoHours2);
            assertIndexEquals(result.indexes, eternity1, eternity2, eternity3);
        }
    }

    private void assertIndexEquals(SnapshotIndex[] actual, SnapshotIndexWithStats... expected) {
        var expect = Stream.of(expected)
            .map(SnapshotIndexWithStats::getIndex)
            .toArray(SnapshotIndex[]::new);

        System.out.println();
        System.out.println("expect: ");
        Stream.of(expect).forEach(System.out::println);
        System.out.println("actual: ");
        Stream.of(actual).forEach(System.out::println);

        assertArrayEquals(expect, actual);
    }

    private MergeAdvice merge(MergeOptions options, MergeKind kind, SnapshotIndexWithStats... source) {
        var strategy = new MergeStrategy(options, options, Clock.systemUTC());
        return strategy.chooseMerge(Stream.of(source), kind);
    }

    private SnapshotIndexWithStats index(SnapshotLevel level, long bytes) {
        return index(level, bytes, System.currentTimeMillis());
    }

    private SnapshotIndexWithStats index(SnapshotLevel level, long bytes, long createdAt) {
        var chunks = generateChunks(bytes);
        var properties = new SnapshotIndexProperties()
            .setCreatedAt(createdAt)
            .setRecordCount(ThreadLocalRandom.current().nextInt(1, 100_000_000))
            .setMetricCount(chunks.metricIdStream().count());
        var content = new SnapshotIndexContent(StockpileFormat.CURRENT, properties, chunks);
        var index = new SnapshotIndex(level, txn++, content);
        var indexSize = new SizeAndCount(1 << 10, 1); // 1 KiB index
        var levelSize = new LevelSizeAndCount(indexSize, content.diskSize(), SizeAndCount.zero);
        return new SnapshotIndexWithStats(index, levelSize);
    }

    private ChunkIndexArray generateChunks(long bytes) {
        ChunkIndexArray result = new ChunkIndexArray();
        var random = ThreadLocalRandom.current();
        int countChunks = 100;
        long bytesAtChunk = bytes / countChunks;
        long localId = StockpileLocalId.random();
        for (int chunkNo = 0; chunkNo < countChunks; chunkNo++) {
            int countMetrics = 500;
            long bytesAtMetric = bytesAtChunk / countMetrics;
            for (int metricNo = 0; metricNo < countMetrics; metricNo++) {
                localId += random.nextInt(1,100);
                result.addMetric(localId, TsRandomData.randomTs(random), Math.toIntExact(bytesAtMetric));
            }
            result.finishChunk();
        }

        if (result.countChunkDiskSize() < bytes) {
            int size = Math.toIntExact(bytes - result.countChunkDiskSize());
            result.addMetric(++localId, TsRandomData.randomTs(random), size);
            result.finishChunk();
        }

        return result;
    }

}
