package ru.yandex.stockpile.server.shard;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.WillNotClose;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.lang.ShortUtils;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DecimPolicyField;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.SummaryDouble;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.shard.MergeProcessMetrics.MergeTaskMetrics;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.model.point.AggrPoints.dhistogram;
import static ru.yandex.solomon.model.point.AggrPoints.lpoint;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.MetricsMatcher.assertEqualTo;
import static ru.yandex.stockpile.server.shard.merge.Utils.assumeMultipleFormats;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeTaskTest {
    private final MergeProcessMetrics metrics = new MergeProcessMetrics();
    private int shardId;
    private long localId;

    @Before
    public void setUp() {
        shardId = StockpileShardId.random();
        localId = StockpileLocalId.random();
    }

    @Test
    public void mergeOneArchiveNoDecim() {
        var source = archive(
            point("2018-04-20T14:53:36Z", 1),
            point("2018-04-20T14:53:46Z", 2),
            point("2018-04-20T14:53:50Z", 3),
            point("2018-04-20T14:53:55Z", 4),
            point("2018-04-20T14:53:59Z", 5));

        long now = System.currentTimeMillis();

        MetricArchiveImmutable daily = merge(MergeKind.DAILY, now, source.toImmutableNoCopy());
        assertEqualTo(source, daily);

        MetricArchiveImmutable eternity = merge(MergeKind.ETERNITY, now, daily);
        assertEqualTo(source, eternity);
        close(source, eternity);
    }

    @Test
    public void collapseSamePoints() {
        var source = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3),
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5));

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {source, source, source};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily, equalTo(source));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(source));
        close(source, daily, eternity);
    }

    @Test
    public void appendPoints() {
        var one = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3));

        var two = archive(
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));

        var tree = archive(
            point("2018-04-20T14:55:00Z", 7));

        var expected = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3),
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6),
            point("2018-04-20T14:55:00Z", 7));

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {one, two, tree};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily, equalTo(expected));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(expected));
        close(one, two, tree, expected, daily, eternity);
    }

    @Test
    public void mergeOwnerShardId() {
        EProjectId projectId = EProjectId.SOLOMON;
        int ownerShardId = -1662362008;

        var one = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3));
        one.setOwnerProjectIdEnum(projectId);
        one.setOwnerShardId(33);

        var two = archive(
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));
        two.setOwnerProjectIdEnum(projectId);
        two.setOwnerShardId(ownerShardId);

        var expected = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3),
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));

        expected.setOwnerProjectIdEnum(projectId);
        expected.setOwnerShardId(ownerShardId);

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {one, two};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertEquals(expected, daily);
        assertEquals(ownerShardId, daily.getOwnerShardId());

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertEquals(expected, eternity);
        assertEquals(ownerShardId, eternity.getOwnerShardId());
        close(one, two, expected, daily, eternity);
    }

    @Test
    public void mergeIntersectingArchives() {
        var one = archive(
            point("2018-04-20T14:52:34Z", 1),
            point("2018-04-20T14:52:44Z", 2),
            point("2018-04-20T14:52:54Z", 3));

        var two = archive(
            point("2018-04-20T14:52:45Z", 2.5),
            point("2018-04-20T14:53:00Z", 4),
            point("2018-04-20T14:53:10Z", 5),
            point("2018-04-20T14:53:10Z", 5),
            point("2018-04-20T14:53:10Z", 5),
            point("2018-04-20T14:53:20Z", 6));

        var tree = archive(
            point("2018-04-20T14:53:10Z", 5),
            point("2018-04-20T14:53:15Z", 5.5),
            point("2018-04-20T14:55:00Z", 7));

        var expected = archive(
            point("2018-04-20T14:52:34Z", 1),
            point("2018-04-20T14:52:44Z", 2),
            point("2018-04-20T14:52:45Z", 2.5),
            point("2018-04-20T14:52:54Z", 3),
            point("2018-04-20T14:53:00Z", 4),
            point("2018-04-20T14:53:10Z", 5),
            point("2018-04-20T14:53:15Z", 5.5),
            point("2018-04-20T14:53:20Z", 6),
            point("2018-04-20T14:55:00Z", 7));

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {one, two, tree};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily, equalTo(expected));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(expected));
        close(one, two, tree, expected, daily, eternity);
    }

    @Test
    public void mergeSameTsLatestWin() {
        var one = archive(
            point("2018-04-20T14:00:00.00Z", 1),
            point("2018-04-20T14:00:00.15Z", 2),
            point("2018-04-20T14:00:00.30Z", 3));

        var two = archive(
            point("2018-04-20T14:00:00.00Z", 42),
            point("2018-04-20T14:00:00.15Z", 42),
            point("2018-04-20T14:00:00.30Z", 42));

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {one, two};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily, equalTo(two));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(two));
        close(one, two, daily, eternity);
    }

    @Test
    public void mergeSameTsCombineAggregates() {
        var one = archive(
            AggrPoint.builder()
                .time("2018-04-20T14:00:00Z")
                .doubleValue(10)
                .merged()
                .count(4)
                .build(),

            AggrPoint.builder()
                .time("2018-04-20T14:00:10Z")
                .doubleValue(8)
                .merged()
                .count(2)
                .build(),

            AggrPoint.builder()
                .time("2018-04-20T14:00:15Z")
                .doubleValue(42)
                .merged()
                .count(15)
                .build()
        );

        var two = archive(
            AggrPoint.builder()
                .time("2018-04-20T14:00:00Z")
                .doubleValue(50)
                .merged()
                .count(11)
                .build(),

            AggrPoint.builder()
                .time("2018-04-20T14:00:10Z")
                .doubleValue(60)
                .merged()
                .count(13)
                .build());


        var expected = archive(
            AggrPoint.builder()
                .time("2018-04-20T14:00:00Z")
                .doubleValue(60)
                .merged()
                .count(15)
                .build(),

            AggrPoint.builder()
                .time("2018-04-20T14:00:10Z")
                .doubleValue(68)
                .merged()
                .count(15)
                .build(),

            AggrPoint.builder()
                .time("2018-04-20T14:00:15Z")
                .doubleValue(42)
                .merged()
                .count(15)
                .build());

        long now = System.currentTimeMillis();
        MetricArchiveMutable[] archives = {one, two};

        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily, equalTo(expected));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(expected));
        close(one, two, expected, daily, eternity);
    }

    @Test
    public void deleteBefore() {
        var one = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3));

        var two = archive(
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));

        var tree = archive(
            point("2018-04-20T14:55:00Z", 7),
            point("2018-04-20T14:55:10Z", 8),
            point("2018-04-20T14:55:20Z", 9));
        tree.setDeleteBefore(timeToMillis("2018-04-20T14:55:00Z"));

        MetricArchiveImmutable expected = tree.toImmutable();

        long now = System.currentTimeMillis();

        MetricArchiveImmutable
            daily = merge(MergeKind.DAILY, now, one.toImmutableNoCopy(), two.toImmutableNoCopy(), tree.toImmutable());
        assertThat(daily, equalTo(expected));

        MetricArchiveImmutable
            eternity = merge(MergeKind.ETERNITY, now, one.toImmutableNoCopy(), two.toImmutableNoCopy(), tree.toImmutable());
        assertThat(eternity, equalTo(expected));
        close(one, two, tree, expected, daily, eternity);
    }

    @Test
    public void deleteAll() {
        var one = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3));

        var two = archive(
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));
        two.setDeleteBefore(DeleteBeforeField.DELETE_ALL);

        var tree = archive(
            point("2018-04-20T14:55:00Z", 7),
            point("2018-04-20T14:55:10Z", 8),
            point("2018-04-20T14:55:20Z", 9));

        MetricArchiveMutable expected = new MetricArchiveMutable();
        expected.setType(MetricType.DGAUGE);
        expected.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        expected.addRecord(point("2018-04-20T14:55:00Z", 7));
        expected.addRecord(point("2018-04-20T14:55:10Z", 8));
        expected.addRecord(point("2018-04-20T14:55:20Z", 9));

        long now = System.currentTimeMillis();

        var daily = merge(MergeKind.DAILY, now, one, two, tree);
        assertThat(daily, equalTo(expected));

        var eternity = merge(MergeKind.ETERNITY, now, one, two, tree);
        assertThat(eternity, equalTo(expected));
        close(one, two, tree, eternity, daily, expected);
    }

    @Test
    public void skipNoCandidateDecim() {
        var one = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3));
        one.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var two = archive(
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));
        two.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var expected = archive(
            point("2018-04-20T14:53:34Z", 1),
            point("2018-04-20T14:53:44Z", 2),
            point("2018-04-20T14:53:54Z", 3),
            point("2018-04-20T14:54:00Z", 4),
            point("2018-04-20T14:54:10Z", 5),
            point("2018-04-20T14:54:20Z", 6));
        expected.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        long now = timeToMillis("2018-04-20T14:54:20Z");
        MetricArchiveImmutable[] archives = {one.toImmutable(), two.toImmutable()};

        MetricArchiveImmutable eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(expected.toImmutable()));
        close(one, two, expected, eternity);
    }

    @Test
    public void decimPoints() {
        MetricArchiveMutable one = archive(
            point("2018-04-20T14:00:00Z", 1),
            point("2018-04-20T14:00:15Z", 2),
            point("2018-04-20T14:00:30Z", 3),
            point("2018-04-20T14:00:45Z", 4),
            point("2018-04-20T14:01:00Z", 5));
        one.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        MetricArchiveMutable two = archive(
            point("2018-04-20T14:05:15Z", 6),
            point("2018-04-20T14:05:30Z", 7),
            point("2018-04-20T14:05:45Z", 8));
        two.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        MetricArchiveMutable expected = archive(
            point("2018-04-20T14:00:00Z", 3),
            point("2018-04-20T14:05:00Z", 7));
        expected.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        long now = timeToMillis("2018-05-01T14:00:00Z");
        MetricArchiveMutable[] archives = {one, two};

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity, equalTo(expected));
        close(one, two, expected, eternity);
    }

    @Test
    public void mergeArchivesWithoutContent() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));
        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        MetricArchiveMutable[] archives = {one, two};

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, archives);
        assertThat(daily.toAggrGraphDataArrayList(), equalTo(MetricArchiveImmutable.empty.toAggrGraphDataArrayList()));

        var eternity = merge(MergeKind.ETERNITY, now, archives);
        assertThat(eternity.toAggrGraphDataArrayList(), equalTo(MetricArchiveImmutable.empty.toAggrGraphDataArrayList()));
        close(one, two, daily, eternity);
    }

    @Test
    public void decimFirstIntoEpochPointsToEmpty() {
        MetricArchiveMutable source = archive(
            point("1970-01-01T00:00:01.000Z", 167),
            point("1970-01-01T00:00:15.000Z", 167));
        source.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));


        long now = System.currentTimeMillis();
        var merged = merge(MergeKind.ETERNITY, now, source);
        var result = merged.toAggrGraphDataArrayList();

        assertThat(result, equalTo(MetricArchiveImmutable.empty.toAggrGraphDataArrayList()));
        close(source, merged);
    }

    @Test
    public void ignoreArchivesWithUnsupportedKindChange() {
        MetricArchiveMutable one = archive(
            MetricType.COUNTER,
            lpoint("2018-04-20T14:05:00Z", 1),
            lpoint("2018-04-20T14:10:00Z", 2),
            lpoint("2018-04-20T14:20:00Z", 3),
            lpoint("2018-04-20T14:30:00Z", 4),
            lpoint("2018-04-20T14:40:00Z", 5));

        MetricArchiveMutable two = archive(
            MetricType.COUNTER,
            lpoint("2018-04-20T14:45:00Z", 6),
            lpoint("2018-05-20T14:50:00Z", 7));

        MetricArchiveMutable tree = archive(
            MetricType.HIST,
            point("2018-05-20T14:50:00Z", dhistogram(new double[]{100, 200, 300}, new long[]{1, 0, 3})));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            lpoint("2018-04-20T14:05:00Z", 1),
            lpoint("2018-04-20T14:10:00Z", 2),
            lpoint("2018-04-20T14:20:00Z", 3),
            lpoint("2018-04-20T14:30:00Z", 4),
            lpoint("2018-04-20T14:40:00Z", 5),
            lpoint("2018-04-20T14:45:00Z", 6),
            lpoint("2018-05-20T14:50:00Z", 7));

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, one, two, tree);
        assertEquals(MetricType.COUNTER, daily.getType());
        assertEquals(StockpileColumns.minColumnSet(MetricType.COUNTER), daily.columnSetMask());
        assertEquals(expected, daily.toAggrGraphDataArrayList());
        close(one, two, tree, daily);
    }

    @Test
    public void migrateFromDGaugeToRateDuringMerge() {
        MetricArchiveMutable one = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:00:00Z", 10d),
            point("2018-01-18T13:00:10Z", 20d),
            point("2018-01-18T13:00:20Z", 50d),
            point("2018-01-18T13:00:30Z", 25d));

        MetricArchiveMutable two = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:00:40Z", 45d),
            point("2018-01-18T13:00:50Z", 3d));

        MetricArchiveMutable tree = archive(
            MetricType.RATE,
            lpoint("2018-01-18T13:01:30Z", 100),
            lpoint("2018-01-18T13:01:45Z", 300),
            lpoint("2018-01-18T13:01:50Z", 840));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            lpoint("2018-01-18T13:00:00Z", 100),
            lpoint("2018-01-18T13:00:10Z", 300),
            lpoint("2018-01-18T13:00:20Z", 800),
            lpoint("2018-01-18T13:00:30Z", 1050),
            lpoint("2018-01-18T13:00:40Z", 1500),
            lpoint("2018-01-18T13:00:50Z", 1530),
            lpoint("2018-01-18T13:01:30Z", 1630),
            lpoint("2018-01-18T13:01:45Z", 1830),
            lpoint("2018-01-18T13:01:50Z", 2370));

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, one, two, tree);
        assertEquals(MetricType.RATE, daily.getType());
        assertEquals(StockpileColumns.minColumnSet(MetricType.RATE), daily.columnSetMask());
        assertEquals(expected, daily.toAggrGraphDataArrayList());
        close(one, two, tree, daily);
    }

    @Test
    public void dropUncompilableKindInTheMiddle() {
        MetricArchiveMutable one = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:00:00Z", 70d),
            point("2018-01-18T13:00:10Z", 20d),
            point("2018-01-18T13:00:20Z", 50d),
            point("2018-01-18T13:00:30Z", 25d));

        MetricArchiveMutable two = archive(
            MetricType.HIST,
            point("2018-01-18T13:00:40Z", dhistogram(new double[]{100, 200, 300}, new long[]{1, 0, 3})),
            point("2018-01-18T13:00:50Z", dhistogram(new double[]{100, 200, 300}, new long[]{5, 1, 2})));

        MetricArchiveMutable tree = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:01:30Z", 100d),
            point("2018-01-18T13:01:45Z", 300d),
            point("2018-01-18T13:01:50Z", 840d));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point("2018-01-18T13:00:00Z", 70d),
            point("2018-01-18T13:00:10Z", 20d),
            point("2018-01-18T13:00:20Z", 50d),
            point("2018-01-18T13:00:30Z", 25d),
            point("2018-01-18T13:01:30Z", 100d),
            point("2018-01-18T13:01:45Z", 300d),
            point("2018-01-18T13:01:50Z", 840d));

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, one, two, tree);
        assertEquals(MetricType.DGAUGE, daily.getType());
        assertEquals(StockpileColumns.minColumnSet(MetricType.DGAUGE), daily.columnSetMask());
        assertEquals(expected, daily.toAggrGraphDataArrayList());
        close(one, two, tree, daily);
    }

    @Test
    public void convertToFinalKind() {
        MetricArchiveMutable one = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:00:00Z", 70d),
            point("2018-01-18T13:00:10Z", 20d),
            point("2018-01-18T13:00:20Z", 50d),
            point("2018-01-18T13:00:30Z", 25d));

        MetricArchiveMutable two = archive(
            MetricType.RATE,
            lpoint("2018-01-18T13:00:40Z", 1000),
            lpoint("2018-01-18T13:00:50Z", 1040));

        MetricArchiveMutable tree = archive(
            MetricType.DGAUGE,
            point("2018-01-18T13:01:30Z", 100d),
            point("2018-01-18T13:01:45Z", 300d),
            point("2018-01-18T13:01:50Z", 840d));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point("2018-01-18T13:00:00Z", 70d),
            point("2018-01-18T13:00:10Z", 20d),
            point("2018-01-18T13:00:20Z", 50d),
            point("2018-01-18T13:00:30Z", 25d),
            // lost caused by integrate
            point("2018-01-18T13:00:50Z", 4d),
            point("2018-01-18T13:01:30Z", 100d),
            point("2018-01-18T13:01:45Z", 300d),
            point("2018-01-18T13:01:50Z", 840d));

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, one, two, tree);
        assertEquals(MetricType.DGAUGE, daily.getType());
        assertEquals(StockpileColumns.minColumnSet(MetricType.DGAUGE), daily.columnSetMask());
        AggrGraphDataArrayList daylyList = daily.toAggrGraphDataArrayList();
        daylyList.foldDenomIntoOne();
        assertEquals(expected, daylyList);
        close(one, two, tree, daily);
    }

    @Test
    public void forceDecim() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        long ts0 = Instant.parse("2018-01-18T13:00:50Z").toEpochMilli();
        MetricArchiveMutable source = new MetricArchiveMutable();
        source.setType(MetricType.DGAUGE);
        AggrPoint point = new AggrPoint();
        point.setTsMillis(ts0);
        source.ensureBytesCapacity(source.columnSetMask(), 250 << 20);
        while (source.bytesCount() <= (250 << 20)) { // <= 250 MiB
            point.tsMillis += random.nextLong(30_000);
            point.setValue(ValueRandomData.randomNum(random), ValueRandomData.randomDenom(random));
            source.addRecord(point);
        }

        long now = System.currentTimeMillis();
        MetricArchiveMutable result = merge(MergeKind.ETERNITY, now, source);
        assertNotEquals(DecimPolicyField.UNDEFINED, result.getDecimPolicyId());
        assertNotEquals(source.getRecordCount(), result.getRecordCount());
        assertNotEquals(source.memorySizeIncludingSelf(), result.memorySizeIncludingSelf());
        close(source, result);
    }

    @Test
    public void mergeAggregateByReplace() {
        long ts0 = System.currentTimeMillis();

        MetricArchiveMutable one = archive(
            MetricType.DGAUGE,
            point(ts0, 42, true, 10));

        MetricArchiveMutable two = archive(
            MetricType.DGAUGE,
            point(ts0, 20, false, 5));

        var expected = AggrGraphDataArrayList.of(point(ts0, 20, false, 5));

        long now = System.currentTimeMillis();
        var daily = merge(MergeKind.DAILY, now, one, two);
        assertEquals(expected, daily.toAggrGraphDataArrayList());
        var eternity = merge(MergeKind.DAILY, now, one, two);
        assertEquals(expected, eternity.toAggrGraphDataArrayList());
        close(one, two, daily, eternity);
    }

    @Test
    public void skipMergeNoDecimNoActive() {
        long ts0 = System.currentTimeMillis();

        var archive = archive(MetricType.DGAUGE, point(ts0, 42, true, 10));
        var immutable = archive.toImmutableNoCopy();

        long now = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(2L);
        long decimatedAt = now - TimeUnit.DAYS.toMillis(1L);

        MetricArchiveImmutable daily = merge(MergeKind.DAILY, now, decimatedAt, immutable);
        assertSame(immutable, daily);

        MetricArchiveImmutable eternity = merge(MergeKind.ETERNITY, now, decimatedAt, immutable);
        assertSame(immutable, eternity);
        close(archive, immutable);
    }

    @Test
    public void skipMergeDecimatedNoActive() {
        long ts0 = System.currentTimeMillis();

        var archive = archive(MetricType.DGAUGE, point(ts0, 42, true, 10));
        archive.setDecimPolicyId((short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());

        { // not decimed yet
            long now = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(2L);
            long decimatedAt = now - TimeUnit.DAYS.toMillis(1L);
            var immutable = archive.toImmutableNoCopy();
            MetricArchiveImmutable daily = merge(MergeKind.DAILY, now, decimatedAt, immutable);
            assertSame(immutable, daily);

            MetricArchiveImmutable eternity = merge(MergeKind.ETERNITY, now, decimatedAt, immutable);
            assertNotSame(immutable, eternity);
            assertEquals(archive.toAggrGraphDataArrayList(), daily.toAggrGraphDataArrayList());
            close(eternity);
        }

        { // already decimated and not active
            long now = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(9L);
            long decimatedAt = now - TimeUnit.DAYS.toMillis(1L);

            var immutable = archive.toImmutableNoCopy();
            MetricArchiveImmutable daily = merge(MergeKind.DAILY, now, decimatedAt, immutable);
            assertSame(immutable, daily);

            MetricArchiveImmutable eternity = merge(MergeKind.ETERNITY, now, decimatedAt, immutable);
            assertSame(immutable, eternity);
            close(immutable);
        }
        close(archive);
    }

    @Test
    public void repack() {
        assumeMultipleFormats();
        var format = StockpileFormat.CURRENT != StockpileFormat.MIN ? StockpileFormat.MIN : StockpileFormat.MAX;
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.addRecord(randomPoint(MetricType.DGAUGE));
        var immutable = archive.toImmutableNoCopy();

        long now = System.currentTimeMillis();
        long decimatedAt = now + TimeUnit.DAYS.toMillis(30);
        var daily = merge(MergeKind.DAILY, now, decimatedAt, immutable);
        assertNotSame(immutable, daily);
        assertEquals(StockpileFormat.CURRENT, daily.getFormat());
        assertEqualTo(archive, daily);

        var immutable2 = archive.toImmutableNoCopy();
        var eternity = merge(MergeKind.ETERNITY, now, decimatedAt, immutable2);
        assertNotSame(immutable2, eternity);
        assertEquals(StockpileFormat.CURRENT, eternity.getFormat());
        assertEqualTo(archive, eternity);
        close(archive, daily, eternity);
    }

    @Test
    public void mergeDecimAgainPolicyChange() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.addAll(AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:17:15Z", 1),
            point("2018-08-01T14:17:30Z", 1),
            point("2018-08-01T14:17:45Z", 1),
            point("2018-08-01T14:21:45Z", 1),
            point("2018-08-01T14:23:45Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setDecimPolicyId((short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        two.addRecord(point("2018-08-09T14:17:15Z", 1));

        var result = merge(MergeKind.ETERNITY, now, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-01T14:23:45Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-09T14:17:15Z"),
                0,
                two.toImmutableNoCopy()
            )
        ));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1),

            // no decim
            point("2018-08-09T14:17:15Z", 1)
        );

        assertEquals(MetricType.DGAUGE, result.getType());
        assertEquals(expected, result.toAggrGraphDataArrayList());
        close(one, two, result);
    }

    @Test
    public void mergeDecimAgainPushIntoPast() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId((short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setDecimPolicyId((short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        two.addAll(AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:17:15Z", 1),
            point("2018-08-01T14:17:30Z", 1),
            point("2018-08-01T14:17:45Z", 1),
            point("2018-08-01T14:21:45Z", 1),
            point("2018-08-01T14:23:45Z", 1),

            // no decim
            point("2018-08-09T14:17:15Z", 1)
        ));
        two.addRecord(point("2018-08-09T14:17:15Z", 1));

        var result = merge(MergeKind.ETERNITY, now, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-01T14:23:45Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-09T14:17:15Z"),
                0,
                two.toImmutableNoCopy()
            )
        ));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1),

            // no decim
            point("2018-08-09T14:17:15Z", 1)
        );

        assertEquals(MetricType.DGAUGE, result.getType());
        assertEquals(expected, result.toAggrGraphDataArrayList());
        close(one, two, result);
    }

    @Test
    public void skipMergeAlreadyDecimated() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        short policy = (short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber();

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId(policy);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setDecimPolicyId(policy);
        two.addRecord(point("2018-08-09T14:17:15Z", 1));

        var result = merge(MergeKind.ETERNITY, now, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-01T14:23:45Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-09T14:17:15Z"),
                0,
                two.toImmutableNoCopy()
            )
        ));

        AggrGraphDataArrayList expectedPoints = AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1),

            // no decim
            point("2018-08-09T14:17:15Z", 1)
        );

        MetricArchiveMutable expectedArchive = new MetricArchiveMutable();
        expectedArchive.setType(MetricType.DGAUGE);
        expectedArchive.setDecimPolicyId(policy);
        expectedArchive.addAll(one);
        expectedArchive.forceCloseFrame();
        expectedArchive.addAll(two);
        expectedArchive.closeFrame();

        assertEquals(MetricType.DGAUGE, result.getType());
        assertEquals(expectedPoints, result.toAggrGraphDataArrayList());
        close(one, two, result, expectedArchive);
    }

    @Test
    public void splitNothing() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        short policy = (short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber();
        long splitDelay = TimeUnit.DAYS.toMillis(30);

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId(policy);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setDecimPolicyId(policy);
        two.addRecord(point("2018-08-09T14:17:15Z", 1));

        var result = split(true, now, splitDelay, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-01T14:23:45Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-09T14:17:15Z"),
                0,
                two.toImmutableNoCopy()
            )
        ));

        MetricArchiveMutable expectedArchive = new MetricArchiveMutable(one);
        expectedArchive.addAll(one);
        expectedArchive.forceCloseFrame();
        expectedArchive.addAll(two);
        expectedArchive.closeFrame();

        {
            var level = result.getCurrentLevel();
            var archive = level.archive();
            assertEquals(MetricType.DGAUGE, archive.getType());
            assertEquals(expectedArchive.getLastTsMillis(), level.lastTsMillis());
            assertEquals(expectedArchive.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            assertEquals(expectedArchive.getCompressedDataRaw(), archive.getCompressedDataRaw());
        }

        {
            assertNull(result.getNextLevel());
        }
        close(one, two, expectedArchive);
    }

    @Test
    public void splitSkip() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        short policy = (short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber();
        long splitDelay = TimeUnit.DAYS.toMillis(30);

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId(policy);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1)
        ));
        one.forceCloseFrame();

        var result = split(true, now, splitDelay, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-01T14:20:00Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            )
        ));

        MetricArchiveMutable expectedArchive = new MetricArchiveMutable(one);
        expectedArchive.addAll(one);
        expectedArchive.closeFrame();

        {
            var level = result.getCurrentLevel();
            var archive = level.archive();
            assertEquals(MetricType.DGAUGE, archive.getType());
            assertEquals(expectedArchive.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            assertEquals(expectedArchive.getCompressedDataRaw(), archive.getCompressedDataRaw());
            assertEquals(timeToMillis("2018-08-01T14:20:00Z"), level.lastTsMillis());
        }

        {
            assertNull(result.getNextLevel());
        }
        close(one, expectedArchive);
    }

    @Test
    public void splitDecimated() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        short policy = (short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber();
        long splitDelay = TimeUnit.DAYS.toMillis(1);

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId(policy);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1),
            point("2018-08-02T14:30:00Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setDecimPolicyId(policy);
        two.addRecord(point("2018-08-09T14:17:15Z", 1));

        var result = split(true, now, splitDelay, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-02T14:30:00Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-09T14:17:15Z"),
                0,
                two.toImmutableNoCopy()
            )
        ));

        {
            var expected = new MetricArchiveMutable(one.header());
            expected.addRecord(point("2018-08-01T14:15:00Z", 1));
            expected.addRecord(point("2018-08-01T14:20:00Z", 1));
            expected.closeFrame();

            var level = result.getNextLevel();
            var archive = level.archive();
            assertEquals(expected.header(), archive.header());
            assertEquals(expected.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            assertEquals(expected.getCompressedDataRaw(), archive.getCompressedDataRaw());
            assertEquals(expected.getLastTsMillis(), level.lastTsMillis());
            close(expected);
        }

        {
            var expected = new MetricArchiveMutable(one.header());
            expected.addRecord(point("2018-08-02T14:30:00Z", 1));
            expected.addRecord(point("2018-08-09T14:17:15Z", 1));
            expected.closeFrame();

            var level = result.getCurrentLevel();
            var archive = level.archive();
            assertEquals(MetricType.DGAUGE, archive.getType());
            assertEquals(expected.getLastTsMillis(), level.lastTsMillis());
            assertEquals(expected.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            assertEquals(expected.getCompressedDataRaw(), archive.getCompressedDataRaw());
            close(expected);
        }
        close(one, two);
    }

    @Test
    public void splitDeleteAll() {
        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        short policy = (short) EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber();
        long splitDelay = TimeUnit.DAYS.toMillis(1);

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setDecimPolicyId(policy);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T14:20:00Z", 1),
            point("2018-08-02T14:30:00Z", 1)
        ));
        one.forceCloseFrame();

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        two.closeFrame();

        var result = split(true, now, splitDelay, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-02T14:30:00Z"),
                timeToMillis("2018-08-08T15:00:00Z"),
                one.toImmutableNoCopy()
            ),
            new MetricIdAndData(
                localId,
                0,
                0,
                two.toImmutableNoCopy()
            )
        ));

        {
            assertNull(result.getCurrentLevel());
        }

        {
            var expected = new MetricArchiveMutable();
            expected.setType(MetricType.DGAUGE);
            expected.setDecimPolicyId(policy);
            expected.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
            expected.closeFrame();

            var level = result.getNextLevel();
            var archive = level.archive();
            assertEquals(expected.header(), archive.header());
            assertEquals(0, level.lastTsMillis());
            assertEquals(AggrGraphDataArrayList.empty(), archive.toAggrGraphDataArrayList());
            close(expected, archive);
        }
        close(one, two);
    }

    @Test
    public void splitWithTimeAlight() {
        long now = timeToMillis("2018-08-02T15:15:18.010Z");
        long delay = TimeUnit.DAYS.toMillis(1);

        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.addAll(AggrGraphDataArrayList.of(
            point("2018-08-01T14:15:00Z", 1),
            point("2018-08-01T20:20:00Z", 1),
            point("2018-08-02T14:30:00Z", 1)
        ));
        one.forceCloseFrame();

        var result = split(false, now, delay, List.of(
            new MetricIdAndData(
                localId,
                timeToMillis("2018-08-02T14:30:00Z"),
                0,
                one.toImmutableNoCopy()
            )
        ));

        {
            var expected = new MetricArchiveMutable(one.header());
            expected.addRecord(point("2018-08-01T14:15:00Z", 1));
            expected.addRecord(point("2018-08-01T20:20:00Z", 1));
            expected.closeFrame();

            var level = result.getNextLevel();
            var archive = level.archive();
            assertEquals(expected.header(), archive.header());
            assertEquals(timeToMillis("2018-08-01T20:20:00Z"), level.lastTsMillis());
            assertEquals(expected.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            close(expected, archive);
        }

        {
            var expected = new MetricArchiveMutable(one.header());
            expected.addRecord(point("2018-08-02T14:30:00Z", 1));
            expected.closeFrame();

            var level = result.getCurrentLevel();
            var archive = level.archive();
            assertEquals(expected.header(), archive.header());
            assertEquals(timeToMillis("2018-08-02T14:30:00Z"), level.lastTsMillis());
            assertEquals(expected.toAggrGraphDataArrayList(), archive.toAggrGraphDataArrayList());
            close(expected, archive);
        }
        close(one);
    }

    @Test
    public void doubleOverflow() {
        var archive = archive(
            point("2018-04-20T14:40:05Z", Double.MAX_VALUE),
            point("2018-04-20T14:40:06Z", Double.MAX_VALUE),
            point("2018-04-20T14:40:07Z", Double.MAX_VALUE),

            point("2018-04-20T14:51:05Z", Double.MAX_VALUE),
            point("2018-04-20T14:51:06Z", 1),
            point("2018-04-20T14:51:07Z", 2));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            point("2018-04-20T14:40:00Z", Double.POSITIVE_INFINITY),
            point("2018-04-20T14:50:00Z", (Double.MAX_VALUE + 1 + 2) / 3));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void igaugeOverflow() {
        var archive = archive(
            lpoint("2018-04-20T14:40:05Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:40:06Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:40:07Z", Long.MAX_VALUE),

            lpoint("2018-04-20T14:51:05Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:51:06Z", 1),
            lpoint("2018-04-20T14:51:07Z", 2));
        archive.setType(MetricType.IGAUGE);
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            lpoint("2018-04-20T14:40:00Z", Long.divideUnsigned(Long.MAX_VALUE + Long.MAX_VALUE + Long.MAX_VALUE, 3)),
            lpoint("2018-04-20T14:50:00Z", (long) Long.divideUnsigned(Long.MAX_VALUE + 1 + 2, 3)));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void counterOverflow() {
        var archive = archive(MetricType.COUNTER,
            lpoint("2018-04-20T14:40:05Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:40:06Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:40:07Z", Long.MAX_VALUE),

            lpoint("2018-04-20T14:51:05Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:51:06Z", 1),
            lpoint("2018-04-20T14:51:07Z", 2));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            lpoint("2018-04-20T14:40:00Z", Long.MAX_VALUE),
            lpoint("2018-04-20T14:50:00Z", 2));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void dSummaryOverflow() {
        var archive = archive(MetricType.DSUMMARY,
            AggrPoints.point("2018-04-20T14:40:05Z", SummaryDouble.newInstance()
                .setCount(Long.MAX_VALUE)
                .setMax(Double.MAX_VALUE)
                .setMin(Double.MAX_VALUE)
                .setSum(Double.MAX_VALUE)
                .setLast(Double.MAX_VALUE)),
            AggrPoints.point("2018-04-20T14:40:06Z", SummaryDouble.newInstance()
                .setCount(10)
                .setMax(Double.MAX_VALUE)
                .setMin(Double.MAX_VALUE)
                .setSum(1)
                .setLast(Double.MAX_VALUE)),
            AggrPoints.point("2018-04-20T14:40:07Z", SummaryDouble.newInstance()
                .setCount(2)
                .setMax(1)
                .setMin(1)
                .setSum(Double.MAX_VALUE - 1)
                .setLast(1)));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            AggrPoints.point("2018-04-20T14:40:00Z", SummaryDouble.newInstance()
                .setCount(Long.MAX_VALUE + 10 + 2)
                .setMax(Double.MAX_VALUE)
                .setMin(1)
                .setSum(Double.POSITIVE_INFINITY)
                .setLast(1)));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void iSummaryOverflow() {
        var archive = archive(MetricType.ISUMMARY,
            AggrPoints.point("2018-04-20T14:40:05Z", new ImmutableSummaryInt64Snapshot(
                Long.MAX_VALUE,
                Long.MAX_VALUE,
                Long.MAX_VALUE,
                Long.MAX_VALUE,
                Long.MAX_VALUE
            )),
            AggrPoints.point("2018-04-20T14:40:06Z", new ImmutableSummaryInt64Snapshot(
                10,
                10,
                10,
                10,
                10
            )),
            AggrPoints.point("2018-04-20T14:40:07Z", new ImmutableSummaryInt64Snapshot(
                2,
                2,
                2,
                2,
                2
            )));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            AggrPoints.point("2018-04-20T14:40:00Z", new ImmutableSummaryInt64Snapshot(
                Long.MAX_VALUE + 10 + 2,
                Long.MAX_VALUE + 10 + 2,
                2,
                Long.MAX_VALUE,
                2
            )));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void histOverflow() {
        var archive = archive(MetricType.HIST,
            AggrPoints.point("2018-04-20T14:40:05Z", Histogram.newInstance()
                .setBucketValue(1, Long.MAX_VALUE)
                .setUpperBound(1, 100)),
            AggrPoints.point("2018-04-20T14:40:06Z", Histogram.newInstance()
                .setBucketValue(1, Long.MAX_VALUE)
                .setUpperBound(1, 100)),
            AggrPoints.point("2018-04-20T14:40:07Z", Histogram.newInstance()
                .setBucketValue(1, 42)
                .setUpperBound(1, 100)));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            AggrPoints.point("2018-04-20T14:40:00Z", Histogram.newInstance()
                .setBucketValue(1, Long.MAX_VALUE)
                .setUpperBound(1, 100)));

        assertEqualTo(expected, result);
        close(archive, result);
    }

    @Test
    public void logHistOverflow() {
        var archive = archive(MetricType.LOG_HISTOGRAM,
            AggrPoints.point("2018-04-20T14:40:05Z", LogHistogram.newInstance()
                .setCountZero(Long.MAX_VALUE)
                .setBuckets(new double[]{Double.MAX_VALUE})
                .build()),
            AggrPoints.point("2018-04-20T14:40:06Z", LogHistogram.newInstance()
                .setCountZero(10)
                .setBuckets(new double[]{Double.MAX_VALUE})
                .build()),
            AggrPoints.point("2018-04-20T14:40:07Z", LogHistogram.newInstance()
                .setCountZero(110)
                .setBuckets(new double[]{2})
                .build()));
        archive.setDecimPolicyId(ShortUtils.toShortExact(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber()));

        var result = merge(MergeKind.ETERNITY, System.currentTimeMillis(), archive);
        var expected = AggrGraphDataArrayList.of(
            AggrPoints.point("2018-04-20T14:40:00Z", LogHistogram.newInstance()
                .setCountZero(Long.MAX_VALUE + 10 + 110)
                .setBuckets(new double[]{Double.POSITIVE_INFINITY})
                .build()));

        assertEqualTo(expected, result);
        close(archive, result);

    }

    private long timeToMillis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private MetricArchiveMutable archive(AggrPoint... points) {
        return MetricArchiveMutable.of(AggrGraphDataArrayList.of(points));
    }

    private MetricArchiveMutable archive(MetricType type, AggrPoint... points) {
        return MetricArchiveMutable.of(type, AggrGraphDataArrayList.of(points));
    }

    private MetricArchiveMutable merge(MergeKind kind, long now, @WillNotClose MetricArchiveMutable... archives) {
        List<MetricIdAndData> sources = Stream.of(archives)
            .map(archive -> {
                archive.sortAndMerge();
                return new MetricIdAndData(localId, archive.getLastTsMillis(), 0, archive.toImmutableNoCopy());
            })
            .collect(Collectors.toList());

        try (var result = merge(kind, now, sources)) {
            return result.toMutable();
        }
    }

    private MetricArchiveImmutable merge(MergeKind kind, long now, MetricArchiveImmutable... archives) {
        return merge(kind, now, 0, archives);
    }

    private MetricArchiveImmutable merge(MergeKind kind, long now, long decimatedAt, MetricArchiveImmutable... archives) {
        List<MetricIdAndData> sources = Stream.of(archives)
            .map(value -> {
                var point = new AggrPoint();
                var it = value.iterator();
                long lastTsMillis = 0L;
                while (it.next(point)) {
                    lastTsMillis = point.tsMillis;
                }

                return new MetricIdAndData(localId, lastTsMillis, decimatedAt, value);
            })
            .collect(Collectors.toList());

        return merge(kind, now, sources);
    }

    private MetricArchiveImmutable merge(MergeKind kind, long now, List<MetricIdAndData> sources) {
        MergeTaskMetrics taskMetrics = metrics.getMergeKindMetrics(kind).getTaskMetrics();
        boolean allowDecim = kind == MergeKind.ETERNITY;
        MergeTask task = new MergeTask(shardId, sources, now, 0, allowDecim, taskMetrics);
        var result = task.run();
        assertNull(result.getNextLevel());
        assertNotNull(result.getCurrentLevel());
        return result.getCurrentLevel().archive();
    }

    private MergeTaskResult split(boolean allowDecim, long now, long splitDelayMillis, List<MetricIdAndData> sources) {
        MergeTaskMetrics taskMetrics = metrics.getMergeKindMetrics(MergeKind.ETERNITY).getTaskMetrics();
        MergeTask task = new MergeTask(shardId, sources, now, splitDelayMillis, allowDecim, taskMetrics);
        return task.run();
    }
}
