package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;
import java.util.StringJoiner;
import java.util.concurrent.Flow;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.SubmissionPublisher;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.shard.iter.SnapshotIterator;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestBase;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class MergeWriterTest extends StockpileShardTestBase {
    @Before
    public void setUp() {
        restart();
    }

    @Test
    public void writeOne() {
        var metric = randomMetric(100);
        var index = write(List.of(metric));
        assertNotNull(index);
        var result = read(index, false);

        assertEquals(1, result.size());
        assertEquals(metric, result.get(0));
    }

    @Test
    public void writeNone() {
        assertNull(write(List.of()));
    }

    @Test
    public void writeMany() {
        List<Metric> source = IntStream.range(0, 500)
            .mapToObj(ignore -> randomMetric(5_000))
            .sorted((o1, o2) -> Long.compareUnsigned(o1.localId, o2.localId))
            .collect(Collectors.toList());

        var index = write(source);
        assertNotNull(index);
        var expected = source.toArray();
        var result = read(index, false).toArray();
        assertArrayEquals(expected, result);
    }

    @Test
    public void memoryUsage() throws InterruptedException {
        var consumer = new MergeWriter(new ShardThreadStub(stockpileShard), System.currentTimeMillis(), SnapshotLevel.DAILY, System.nanoTime(), 42, new MergeProcessMetrics().getMergeKindMetrics(MergeKind.DAILY));
        var initMemoryUsage = consumer.memorySizeIncludingSelf();
        assertNotEquals(0, initMemoryUsage);

        var publisher = new SubmissionPublisher<MetricIdAndData>();
        publisher.subscribe(consumer);
        assertEquals(initMemoryUsage, consumer.memorySizeIncludingSelf());

        publisher.submit(randomData(1, 0));
        publisher.submit(randomData(2, 1));
        publisher.submit(randomData(3, 0));

        while (publisher.estimateMaximumLag() != 0) {
            TimeUnit.MILLISECONDS.sleep(1);
        }
        assertThat(consumer.memorySizeIncludingSelf(), Matchers.greaterThan(initMemoryUsage));
    }

    private MetricIdAndData randomData(long localId, int countPoints) {
        MetricIdAndData data = randomMetric(countPoints).prepare().getCurrentLevel();
        return new MetricIdAndData(localId, data.lastTsMillis(), data.decimatedAt(), data.archive());
    }

    private Metric randomMetric(int countPoints) {
        long localId = StockpileLocalId.random();
        AggrPoint point = new AggrPoint();
        long now = System.currentTimeMillis();
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);

        for (int pointIndex = 0; pointIndex < countPoints; pointIndex++) {
            now += 10_000;
            point.setTsMillis(now);
            point.setValue(ValueRandomData.randomNum(ThreadLocalRandom.current()));
            archive.addRecord(point);
        }

        return new Metric(localId, archive);
    }

    @Nullable
    private SnapshotIndex write(List<Metric> source) {
        var shardThread = new ShardThreadStub(stockpileShard);
        var txn = 42;
        var metrics = new MergeProcessMetrics();
        var kind = MergeKind.DAILY;
        var now = System.currentTimeMillis();

        var producer = new Publisher(source);
        var consumer = new MergeWriter(shardThread, now, kind.targetLevel, now, txn, metrics.getMergeKindMetrics(kind));
        var initMemoryUsage = consumer.memorySizeIncludingSelf();
        assertThat(initMemoryUsage, Matchers.greaterThan(0L));
        var filter = new MergeFilter(MergeTaskResult::getCurrentLevel, false);
        filter.subscribe(consumer);
        producer.subscribe(filter);
        var written = consumer.getDoneFuture().join().orElse(null);
        if (written == null) {
            return null;
        }

        stockpileShard.storage.renameSnapshotDeleteOld(
            new SnapshotAddress[]{written.address()}, new SnapshotAddress[0])
            .join();

        return written.getIndex();
    }

    private List<Metric> read(SnapshotIndex index, boolean onlyIds) {
        var shardThread = new ShardThreadStub(stockpileShard);
        var result = new ArrayList<Metric>();
        var iterator = new SnapshotIterator(stockpileShard.storage, shardThread, index, ReadClass.OTHER);
        while (true) {
            var metric = iterator.next().join();
            if (metric == null) {
                return result;
            }
            MetricArchiveMutable archive = onlyIds ? new MetricArchiveMutable() : metric.archive().toMutable();
            result.add(new Metric(metric.localId(), archive));
        }
    }

    private static class Metric {
        private final long localId;
        private final MetricArchiveMutable archive;

        public Metric(long localId, MetricArchiveMutable archive) {
            this.localId = localId;
            this.archive = archive;
        }

        public MergeTaskResult prepare() {
            var current = new MetricIdAndData(localId, archive.getLastTsMillis(), 0, archive.toImmutableNoCopy());
            return new MergeTaskResult(current);
        }

        public Metric duplicate(long localId) {
            return new Metric(localId, archive);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Metric metric = (Metric) o;

            if (localId != metric.localId) return false;
            return archive.equals(metric.archive);
        }

        @Override
        public int hashCode() {
            int result = (int) (localId ^ (localId >>> 32));
            result = 31 * result + archive.hashCode();
            return result;
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", Metric.class.getSimpleName() + "[", "]")
                .add("localId=" + localId)
                .add("archive=" + archive)
                .toString();
        }
    }

    private static class Publisher implements Flow.Publisher<MergeTaskResult> {
        private AtomicLong requested = new AtomicLong();
        private Flow.Subscriber<? super MergeTaskResult> subscriber;
        private ActorRunner actor;
        private int index = 0;
        private List<Metric> metrics;
        private boolean doneCalled = false;

        public Publisher(List<Metric> metrics) {
            this.metrics = metrics;
            actor = new ActorRunner(this::act, ForkJoinPool.commonPool(), throwable -> subscriber.onError(throwable));
        }

        @Override
        public void subscribe(Flow.Subscriber<? super MergeTaskResult> subscriber) {
            this.subscriber = subscriber;
            subscriber.onSubscribe(new Flow.Subscription() {
                @Override
                public void request(long n) {
                    requested.addAndGet(n);
                    actor.schedule();
                }

                @Override
                public void cancel() {
                    requested.set(0);
                    actor.schedule();
                }
            });
        }

        private void act() {
            if (requested.get() <= 0 || doneCalled) {
                return;
            }

            while (requested.get() > 0) {
                if (index == metrics.size()) {
                    doneCalled = true;
                    subscriber.onComplete();
                    return;
                }

                requested.decrementAndGet();
                var metric = metrics.get(index++);
                subscriber.onNext(metric.prepare());
            }
        }
    }
}
