package ru.yandex.stockpile.server.shard;

import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.LongFileTypeSnapshotCommand;
import ru.yandex.stockpile.server.data.command.SnapshotCommandContent;
import ru.yandex.stockpile.server.data.command.SnapshotCommandPartsSerialized;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestBase;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assume.assumeTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotCommandLoaderImplTest extends StockpileShardTestBase {

    private SnapshotCommandLoaderImpl loader;

    @Before
    public void setUp() {
        assumeTrue("Unsupported format", StockpileFormat.CURRENT.ge(StockpileFormat.DELETED_SHARDS_39));
        restart();
        loader = new SnapshotCommandLoaderImpl(stockpileShard.stateDone());
    }

    @Test
    public void empty() {
        var loaded = loader.readSnapshot(new SnapshotAddress(SnapshotLevel.TWO_HOURS, 42)).join();
        assertEquals(Optional.empty(), loaded);
    }

    @Test
    public void onePartCommand() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 32);
        var content = new SnapshotCommandContent(deletedShards);
        var address = new SnapshotAddress(SnapshotLevel.DAILY, 42);
        writeSnapshot(address, content, Integer.MAX_VALUE);

        var loaded = loader.readSnapshot(address).join();
        assertFalse(loaded.isEmpty());
        assertEquals(address, loaded.get().snapshotAddress());
        assertEquals(deletedShards, loaded.get().content().deletedShards());
    }

    @Test
    public void multiPartCommand() {
        var deletedShards = new DeletedShardSet();
        var random = ThreadLocalRandom.current();
        for (int index = 0; index < 1000; index++) {
            deletedShards.add(random.nextInt(3), random.nextInt());
        }
        var content = new SnapshotCommandContent(deletedShards);
        var address = new SnapshotAddress(SnapshotLevel.DAILY, 42);
        writeSnapshot(address, content, 1000);

        var loaded = loader.readSnapshot(address).join();
        assertFalse(loaded.isEmpty());
        assertEquals(address, loaded.get().snapshotAddress());
        assertEquals(deletedShards, loaded.get().content().deletedShards());
    }

    @Test
    public void readBySnapshotAddress() {
        final SnapshotCommandContent one;
        {
            var deletedShards = new DeletedShardSet();
            deletedShards.add(1, 2);
            one = new SnapshotCommandContent(deletedShards);
        }
        var addressOne = new SnapshotAddress(SnapshotLevel.TWO_HOURS, 5);
        writeSnapshot(addressOne, one, Integer.MAX_VALUE);

        final SnapshotCommandContent two;
        {
            var deletedShards = new DeletedShardSet();
            deletedShards.add(3, 4);
            two = new SnapshotCommandContent(deletedShards);
        }
        var addressTwo = new SnapshotAddress(SnapshotLevel.TWO_HOURS, 6);
        writeSnapshot(addressTwo, two, Integer.MAX_VALUE);

        var loadOne = loader.readSnapshot(addressOne).join();
        assertFalse(loadOne.isEmpty());
        assertEquals(addressOne, loadOne.get().snapshotAddress());
        assertEquals(one.deletedShards(), loadOne.get().content().deletedShards());

        var loadTwo = loader.readSnapshot(addressTwo).join();
        assertFalse(loadTwo.isEmpty());
        assertEquals(addressTwo, loadTwo.get().snapshotAddress());
        assertEquals(two.deletedShards(), loadTwo.get().content().deletedShards());
    }

    private void writeSnapshot(SnapshotAddress address, SnapshotCommandContent content, int splitSize) {
        var huge = LongFileTypeSnapshotCommand.I.serializer().serializeToByteString(content);
        var parts = ByteStrings.split(huge, splitSize);
        var serialized = new SnapshotCommandPartsSerialized(address.level(), address.txn(), parts);
        stockpileShard.storage.writeSnapshotCommandToTemp(serialized).join();
        stockpileShard.storage.renameSnapshotDeleteLogs(serialized.snapshotAddress()).join();
    }
}
