package ru.yandex.stockpile.server.shard;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.SnapshotCommandContent;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestBase;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotCommandWriterTest extends StockpileShardTestBase {

    private SnapshotCommandLoaderImpl loader;
    private SnapshotCommandWriter writer;

    @Before
    public void setUp() {
        assumeTrue("Unsupported format", StockpileFormat.CURRENT.ge(StockpileFormat.DELETED_SHARDS_39));
        restart();
        loader = new SnapshotCommandLoaderImpl(stockpileShard.stateDone());
        writer = new SnapshotCommandWriter(stockpileShard.stateDone());
    }

    @Test
    public void skipWriteEmpty() {
        var content = new SnapshotCommandContent(new DeletedShardSet());
        assertFalse(SnapshotCommandWriter.isWritable(false, content));
    }

    @Test
    public void skipWriteForLatestSnapshot() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 2);
        var content = new SnapshotCommandContent(deletedShards);
        assertFalse(SnapshotCommandWriter.isWritable(true, content));
    }

    @Test
    public void write() {
        var deletedShards = new DeletedShardSet();
        deletedShards.add(1, 2);

        var content = new SnapshotCommandContent(deletedShards);
        var address = new SnapshotAddress(SnapshotLevel.ETERNITY, 42);
        assertTrue(SnapshotCommandWriter.isWritable(false, content));
        writer.write(address.level(), address.txn(), content).join();
        stockpileShard.storage.renameSnapshotDeleteLogs(address).join();

        var loaded = loader.readSnapshot(address).join();
        assertFalse(loaded.isEmpty());
        assertEquals(address, loaded.get().snapshotAddress());
        assertEquals(deletedShards, loaded.get().content().deletedShards());
    }
}
