package ru.yandex.stockpile.server.shard;

import java.util.ArrayList;
import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.random.Random2;

/**
 * @author Stepan Koltsov
 */
public class TxTrackerTest {

    private long ts = 0;

    private class TestTx implements TxTracker.Tx {
        private final long txn;

        public TestTx(long txn) {
            this.txn = txn;
        }

        @Override
        public long txn() {
            return txn;
        }

        private boolean completed = false;
        private long completedTs;

        @Override
        public void completeTx() {
            Assert.assertFalse(completed);
            completed = true;
            completedTs = ++ts;
        }
    }

    @Test
    public void simpleSync() {
        TxTracker txTracker = new TxTracker(10);

        for (long txn = 11; txn < 20; ++txn) {
            Assert.assertEquals(txn, txTracker.allocateTx());
            TestTx tx = new TestTx(txn);
            txTracker.written(tx);
            Assert.assertTrue(tx.completed);
        }
    }

    @Test
    public void outOfOrder() {
        TxTracker txTracker = new TxTracker(10);

        TestTx tx1 = new TestTx(txTracker.allocateTx());
        TestTx tx2 = new TestTx(txTracker.allocateTx());
        Assert.assertEquals(11, tx1.txn());
        Assert.assertEquals(12, tx2.txn());

        txTracker.written(tx2);

        Assert.assertFalse(tx1.completed);
        Assert.assertFalse(tx2.completed);

        txTracker.written(tx1);

        Assert.assertTrue(tx1.completed);
        Assert.assertTrue(tx2.completed);

        Assert.assertEquals(1, tx1.completedTs);
        Assert.assertEquals(2, tx2.completedTs);
    }

    @Test
    public void random() {

        Random2 random = new Random2(17);

        TxTracker txTracker = new TxTracker(1000);

        ArrayList<TestTx> txs = new ArrayList<>();
        for (int i = 0; i < 100; ++i) {
            TestTx tx = new TestTx(txTracker.allocateTx());
            Assert.assertEquals(1001 + i, tx.txn);
            txs.add(tx);
        }

        List<TestTx> shuffled = random.shuffle(txs);
        for (TestTx testTx : shuffled) {
            txTracker.written(testTx);
        }

        for (int i = 0; i < 100; ++i) {
            TestTx tx = txs.get(i);
            Assert.assertTrue(tx.completed);
            Assert.assertEquals(i + 1, tx.completedTs);
        }
    }

}
