package ru.yandex.stockpile.server.shard.cache;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.ZoneOffset;
import java.util.Arrays;

import org.junit.Test;

import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.server.shard.merge.MetricsMatcher;

import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class CacheMetricArchiveMutableTest {

    @Test
    public void emptyArchive() {
        var archive = archive();
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(100, 200));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(100, System.currentTimeMillis()));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(System.currentTimeMillis(), System.currentTimeMillis() + 15_000));
        close(archive);
    }

    @Test
    public void closedFull() {
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2),
            point(ts("00:01:30"), 3));

        var archive = archive();
        archive.addAll(expected);
        archive.forceCloseFrame();

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:00:30"), ts("00:05:00")));
        assertEqualTo(expected, archive.snapshot(0, 0));
        close(archive);
    }

    @Test
    public void latestFull() {
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2),
            point(ts("00:01:30"), 3));

        var archive = archive();
        archive.addAll(expected);

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:00:30"), ts("00:05:00")));
        assertEqualTo(expected, archive.snapshot(0, 0));
        close(archive);
    }

    @Test
    public void latestFullUnsorted() {
        var archive = archive();
        archive.addAllNoSortMergeFrom(AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 3),
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2)
        ).iterator());

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2),
            point(ts("00:01:30"), 3)
        );

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("10:00:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:00:30"), ts("00:05:00")));
        assertEqualTo(expected, archive.snapshot(0, 0));
        close(archive);
    }

    @Test
    public void latestCrop() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        ));
        archive.forceCloseFrame();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:35"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        ));

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:02:00"), ts("00:03:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:35"), 4)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:39")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:40")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:35"), ts("00:01:40")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:35"), 4),
                point(ts("00:01:40"), 5)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:49")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:50")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:35"), ts("00:01:50")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:40"), 5)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:36"), ts("00:01:49")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:36"), ts("00:01:50")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:40"), ts("00:01:50")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:50"), 6)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:45"), ts("00:02:00")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:50"), ts("00:02:00")));
        }
        close(archive);
    }

    @Test
    public void latestCropUnsorted() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        ));
        archive.forceCloseFrame();
        archive.addAllNoSortMergeFrom(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:40"), 5),
            point(ts("00:01:35"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        ).iterator());

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:02:00"), ts("00:03:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:35"), 4)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:39")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:40")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:35"), ts("00:01:40")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:35"), 4),
                point(ts("00:01:40"), 5)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:49")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:50")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:35"), ts("00:01:50")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:40"), 5)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:36"), ts("00:01:49")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:36"), ts("00:01:50")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:40"), ts("00:01:50")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:50"), 6)
            );
            assertEqualTo(expected, archive.snapshot(ts("00:01:45"), ts("00:02:00")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:50"), ts("00:02:00")));
        }
        close(archive);
    }

    @Test
    public void closedFullMultiple() {
        AggrGraphDataArrayList one = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        );

        AggrGraphDataArrayList two = AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        );

        AggrGraphDataArrayList tree = AggrGraphDataArrayList.of(
            point(ts("00:02:00"), 7),
            point(ts("00:02:10"), 8),
            point(ts("00:02:20"), 9)
        );

        var expected = new AggrGraphDataArrayList();
        var archive = archive();

        for (var item : Arrays.asList(one, two, tree)) {
            archive.addAll(item);
            archive.forceCloseFrame();
            expected.addAll(item);
        }

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:03:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:03:00")));
        close(archive);
    }

    @Test
    public void fullMultiple() {
        AggrGraphDataArrayList one = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        );

        AggrGraphDataArrayList two = AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        );

        AggrGraphDataArrayList tree = AggrGraphDataArrayList.of(
            point(ts("00:02:00"), 7),
            point(ts("00:02:10"), 8),
            point(ts("00:02:20"), 9)
        );

        var expected = new AggrGraphDataArrayList();
        var archive = archive();

        for (var item : Arrays.asList(one, two, tree)) {
            archive.addAll(item);
            if (item != tree) {
                archive.forceCloseFrame();
            }
            expected.addAll(item);
        }

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:03:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:03:00")));
        close(archive);
    }

    @Test
    public void fullMultipleUnsorted() {
        AggrGraphDataArrayList one = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        );

        AggrGraphDataArrayList two = AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        );

        AggrGraphDataArrayList tree = AggrGraphDataArrayList.of(
            // replacements
            point(ts("00:01:00"), 2),
            point(ts("00:01:40"), 6),

            // unsorted
            point(ts("00:02:20"), 9),
            point(ts("00:02:00"), 7),
            point(ts("00:02:10"), 8)
        );

        var expected = new AggrGraphDataArrayList();
        var archive = archive();

        for (var item : Arrays.asList(one, two, tree)) {
            archive.addAllNoSortMergeFrom(item.iterator());
            if (item != tree) {
                archive.forceCloseFrame();
            }
            expected.addAll(item);
        }
        expected.sortAndMerge();

        assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:03:00")));
        assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:03:00")));
        close(archive);
    }

    @Test
    public void closedOneCrop() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        close(archive);
    }

    @Test
    public void lastOneCrop() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        close(archive);
    }

    @Test
    public void latestOneCropUnsorted() {
        var archive = archive();
        archive.addAllNoSortMergeFrom(AggrGraphDataArrayList.of(
            point(ts("00:01:20"), 3),
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2)
        ).iterator());

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        close(archive);
    }

    @Test
    public void closedMultipleCrop() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)));
        archive.forceCloseFrame();

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:32")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:40")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:55")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:29"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:55")));
        }
        close(archive);
    }

    @Test
    public void multipleCrop() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)));

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 2));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:32")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:40")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 2),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:55")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:29"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:55")));
        }
        close(archive);
    }

    @Test
    public void multipleCropLastUnsorted() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();
        archive.addAllNoSortMergeFrom(AggrGraphDataArrayList.of(
            point(ts("00:01:40"), 5),
            point(ts("00:01:30"), 4),
            point(ts("00:01:50"), 6),
            point(ts("00:01:10"), 7)).iterator());

        {
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
            assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:00"), 1));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:09")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:10")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:10")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:00"), 1),
                point(ts("00:01:10"), 7));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:00"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:00:00"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(point(ts("00:01:10"), 7));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:19")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:20")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:20")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 7),
                point(ts("00:01:20"), 3));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:29")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:30")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 7),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:32")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:40")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:40")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:10"), 7),
                point(ts("00:01:20"), 3),
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:09"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:10"), ts("00:01:55")));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                point(ts("00:01:30"), 4),
                point(ts("00:01:40"), 5),
                point(ts("00:01:50"), 6));
            assertEqualTo(expected, archive.snapshot(ts("00:01:29"), ts("00:01:55")));
            assertEqualTo(expected, archive.snapshot(ts("00:01:30"), ts("00:01:55")));
        }
        close(archive);
    }

    @Test
    public void closedMultipleGap() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:30:30"), 14),
            point(ts("00:31:40"), 15),
            point(ts("00:31:50"), 16)));
        archive.forceCloseFrame();

        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:10:00"), ts("00:30:00")));
        close(archive);
    }

    @Test
    public void multipleGap() {
        var archive = archive();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)));
        archive.forceCloseFrame();
        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:30:30"), 14),
            point(ts("00:31:40"), 15),
            point(ts("00:31:50"), 16)));

        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:00:00"), ts("00:01:00")));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:40:00"), ts("00:50:00")));
        assertEqualTo(AggrGraphDataArrayList.empty(), archive.snapshot(ts("00:10:00"), ts("00:30:00")));
        close(archive);
    }

    @Test
    public void closedSafeSnapshot() {
        var expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3));

        var archive = archive();
        archive.addAll(expected);
        archive.forceCloseFrame();

        var snapshot = archive.snapshot(ts("00:00:00"), ts("10:00:00"));

        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 4),
            point(ts("00:01:10"), 5),
            point(ts("00:01:20"), 6),
            point(ts("00:01:30"), 7)));
        archive.sortAndMerge();

        assertEqualTo(expected, snapshot);
        close(archive);
    }

    @Test
    public void closedMultipleSafeSnapshot() {
        AggrGraphDataArrayList one = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:10"), 2),
            point(ts("00:01:20"), 3)
        );

        AggrGraphDataArrayList two = AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 4),
            point(ts("00:01:40"), 5),
            point(ts("00:01:50"), 6)
        );

        AggrGraphDataArrayList tree = AggrGraphDataArrayList.of(
            point(ts("00:02:00"), 7),
            point(ts("00:02:10"), 8),
            point(ts("00:02:20"), 9)
        );

        var expected = new AggrGraphDataArrayList();
        var archive = archive();

        for (var item : Arrays.asList(one, two, tree)) {
            archive.addAll(item);
            archive.forceCloseFrame();
            expected.addAll(item);
        }

        var snapshot = archive.snapshot(ts("00:00:00"), ts("00:03:00"));

        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 8),
            point(ts("00:01:40"), 9),
            point(ts("00:01:45"), 10),
            point(ts("00:01:50"), 11),
            point(ts("00:01:50"), 12),
            point(ts("00:02:10"), 13),
            point(ts("00:03:00"), 14)));
        archive.sortAndMerge();

        assertEqualTo(expected, snapshot);
        close(archive);
    }

    @Test
    public void lastSortedSafeSnapshot() {
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2),
            point(ts("00:01:30"), 3));

        var archive = archive();
        archive.addAll(expected);

        var snapshot = archive.snapshot(ts("00:00:00"), ts("10:00:00"));

        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:40"), 4)
        ));
        archive.forceCloseFrame();
        assertEqualTo(expected, snapshot);
        close(archive);
    }

    @Test
    public void lastUnsortedSafeSnapshot() {
        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2),
            point(ts("00:01:30"), 3));

        var archive = archive();
        archive.addAllNoSortMergeFrom(AggrGraphDataArrayList.of(
            point(ts("00:01:30"), 3),
            point(ts("00:01:00"), 1),
            point(ts("00:01:15"), 2))
            .iterator());

        var snapshot = archive.snapshot(ts("00:00:00"), ts("10:00:00"));

        archive.addAll(AggrGraphDataArrayList.of(
            point(ts("00:01:40"), 4)
        ));
        archive.sortAndMerge();
        archive.forceCloseFrame();
        assertEqualTo(expected, snapshot);
        close(archive);
    }

    private CacheMetricArchiveMutable archive() {
        var result = new CacheMetricArchiveMutable();
        result.setType(MetricType.DGAUGE);
        result.ensureCapacity(TsColumn.mask | ValueColumn.mask, 3);
        return result;
    }

    private static long ts(String time) {
        return LocalDateTime.of(LocalDate.of(2000, Month.JANUARY, 1), LocalTime.parse(time))
            .toInstant(ZoneOffset.UTC)
            .toEpochMilli();
    }

    public static void assertEqualTo(AggrGraphDataIterable expect, MetricSnapshot actual) {
        MetricsMatcher.assertEqualTo(expect.iterator(), actual.iterator());
        actual.release();
    }
}
