package ru.yandex.stockpile.server.shard.cache;

import java.util.Arrays;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Stepan Koltsov
 */
public class CacheWeightsTest {


    @Test
    public void sameWeightOneKey() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeightsOneGroup(
            Arrays.asList(shards),
            s -> 44,
            6,
            sizes
        );

        Assert.assertArrayEquals(new long[] { 2, 2, 2 }, sizes);
    }

    @Test
    public void oneKeyAllWeightToOne() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeightsOneGroup(
            Arrays.asList(shards),
            s -> s.equals("b") ? 10 : 0,
            15,
            sizes);
        Assert.assertArrayEquals(new long[] { 0, 15, 0 }, sizes);
    }

    @Test
    public void unequalWeights() {
        String[] shards = new String[] { "a", "bb", "ccc" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeightsOneGroup(
            Arrays.asList(shards),
            String::length,
            18,
            sizes);
        Assert.assertArrayEquals(new long[] { 3, 6, 9 }, sizes);
    }

    private enum Two { ONE, TWO }

    @Test
    public void twoKeysSameWeight() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeights(
            Arrays.asList(shards),
            Two.class,
            o -> 22,
            (s, one) -> 33,
            12,
            sizes);

        Assert.assertArrayEquals(new long[] { 4, 4, 4 }, sizes);
    }

    @Test
    public void twoKeysZerosByOneKey() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeights(
            Arrays.asList(shards),
            Two.class,
            o -> 1,
            (s, two) -> two == Two.ONE ? 100 : 0,
            33,
            sizes);

        Assert.assertArrayEquals(new long[] { 11, 11, 11 }, sizes);
    }

    @Test
    public void twoKeysMultipleZeros() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeights(
            Arrays.asList(shards),
            Two.class,
            t -> t.ordinal() + 1,
            (s, two) -> {
                if (two == Two.ONE && s.equals("a")) {
                    return 10000;
                } else if (two == Two.TWO && s.equals("b")) {
                    return 4444;
                } else {
                    return 0;
                }
            },
            9,
            sizes);

        Assert.assertArrayEquals(new long[] { 3, 6, 0 }, sizes);
    }

    @Test
    public void overflow() {
        String[] shards = new String[] { "a", "b", "c" };
        long[] sizes = new long[shards.length];
        CacheWeights.updateWeightsOneGroup(
            Arrays.asList(shards),
            s -> 10_000_000_000L,
            30_000_000_000L,
            sizes);
        Assert.assertArrayEquals(new long[] { 10_000_000_000L, 10_000_000_000L, 10_000_000_000L }, sizes);
    }
}
